#!/usr/bin/env node
/**
 * List calendar events
 * 
 * Usage:
 *   node list-events.mjs
 *   node list-events.mjs --start "2026-02-01" --end "2026-02-28"
 */

import { parseArgs } from 'util';
import { listEvents, timestampToDatetime, DEFAULT_CALENDAR_ID, DEFAULT_TIMEZONE } from '../lib/calendar.mjs';

const { values } = parseArgs({
  options: {
    start: { type: 'string' },
    end: { type: 'string' },
    timezone: { type: 'string', default: DEFAULT_TIMEZONE },
    calendar: { type: 'string', default: DEFAULT_CALENDAR_ID },
    json: { type: 'boolean' },
    help: { type: 'boolean', short: 'h' }
  }
});

if (values.help) {
  console.log(`
List Lark calendar events

Options:
  --start      Start date: YYYY-MM-DD (default: today)
  --end        End date: YYYY-MM-DD (default: 7 days from now)
  --timezone   IANA timezone (default: Asia/Singapore)
  --calendar   Calendar ID
  --json       Output as JSON
  -h, --help   Show this help

Examples:
  node list-events.mjs
  node list-events.mjs --start "2026-02-01" --end "2026-02-28"
`);
  process.exit(0);
}

async function main() {
  try {
    const events = await listEvents({
      calendarId: values.calendar,
      startTime: values.start,
      endTime: values.end,
      timezone: values.timezone
    });
    
    if (values.json) {
      console.log(JSON.stringify(events, null, 2));
      return;
    }
    
    if (events.length === 0) {
      console.log('No events found in the specified range.');
      return;
    }
    
    console.log(`Found ${events.length} event(s):\n`);
    
    for (const event of events) {
      const startTime = event.start_time?.timestamp 
        ? timestampToDatetime(parseInt(event.start_time.timestamp), values.timezone)
        : 'N/A';
      const endTime = event.end_time?.timestamp
        ? timestampToDatetime(parseInt(event.end_time.timestamp), values.timezone)
        : 'N/A';
      
      console.log(`📅 ${event.summary || '(No title)'}`);
      console.log(`   ID: ${event.event_id}`);
      console.log(`   Time: ${startTime} → ${endTime}`);
      if (event.location?.name) {
        console.log(`   Location: ${event.location.name}`);
      }
      if (event.description) {
        console.log(`   Description: ${event.description.substring(0, 100)}${event.description.length > 100 ? '...' : ''}`);
      }
      console.log('');
    }
    
  } catch (error) {
    console.error('❌ Failed to list events:', error.message);
    if (error.larkResponse) {
      console.error('Lark response:', JSON.stringify(error.larkResponse, null, 2));
    }
    process.exit(1);
  }
}

main();
