#!/usr/bin/env node
/**
 * Delete a calendar event
 * 
 * Usage:
 *   node delete-event.mjs --event-id "f9900f6b-b472-4b17-a818-7b5584abdc37_0"
 */

import { parseArgs } from 'util';
import { deleteEvent, DEFAULT_CALENDAR_ID } from '../lib/calendar.mjs';

const { values } = parseArgs({
  options: {
    'event-id': { type: 'string' },
    calendar: { type: 'string', default: DEFAULT_CALENDAR_ID },
    'no-notify': { type: 'boolean', default: false },
    help: { type: 'boolean', short: 'h' }
  }
});

if (values.help) {
  console.log(`
Delete a Lark calendar event

Options:
  --event-id    Event ID to delete (required)
  --calendar    Calendar ID
  --no-notify   Don't send notifications to attendees
  -h, --help    Show this help

Examples:
  node delete-event.mjs --event-id "f9900f6b-b472-4b17-a818-7b5584abdc37_0"
`);
  process.exit(0);
}

if (!values['event-id']) {
  console.error('Error: --event-id is required');
  process.exit(1);
}

async function main() {
  try {
    console.log(`Deleting event: ${values['event-id']}`);
    
    await deleteEvent(values['event-id'], values.calendar, !values['no-notify']);
    
    console.log('✅ Event deleted successfully!');
    
  } catch (error) {
    console.error('❌ Failed to delete event:', error.message);
    if (error.larkResponse) {
      console.error('Lark response:', JSON.stringify(error.larkResponse, null, 2));
    }
    process.exit(1);
  }
}

main();
