#!/usr/bin/env node
/**
 * Create a task
 * 
 * Usage:
 *   node create-task.mjs --title "Review PR" --due "2026-02-05 18:00:00"
 */

import { parseArgs } from 'util';
import { createTask } from '../lib/task.mjs';
import { DEFAULT_TIMEZONE } from '../lib/calendar.mjs';
import { resolveNames, getDisplayNameSync } from '../lib/employees.mjs';

const { values } = parseArgs({
  options: {
    title: { type: 'string' },
    description: { type: 'string', default: '' },
    due: { type: 'string' },
    assignees: { type: 'string', default: '' },
    'assignee-ids': { type: 'string', default: '' },
    timezone: { type: 'string', default: DEFAULT_TIMEZONE },
    help: { type: 'boolean', short: 'h' }
  }
});

if (values.help) {
  console.log(`
Create a Lark task

Options:
  --title         Task title (required)
  --description   Task description
  --due           Due time: YYYY-MM-DD HH:MM:SS (required)
  --assignees     Comma-separated names (auto-resolved to user_ids)
  --assignee-ids  Comma-separated user_ids directly
  --timezone      IANA timezone (default: Asia/Singapore)
  -h, --help      Show this help

Examples:
  node create-task.mjs --title "Review PR #123" --due "2026-02-05 18:00:00"
  node create-task.mjs --title "Finish report" --due "2026-02-03 17:00:00" --assignees "Boyang,jc"
`);
  process.exit(0);
}

// Validate required fields
if (!values.title) {
  console.error('Error: --title is required');
  process.exit(1);
}
if (!values.due) {
  console.error('Error: --due is required');
  process.exit(1);
}

// Resolve assignees
let assigneeIds = [];
if (values['assignee-ids']) {
  assigneeIds = values['assignee-ids'].split(',').map(s => s.trim()).filter(Boolean);
}
if (values.assignees) {
  const names = values.assignees.split(',').map(s => s.trim()).filter(Boolean);
  const { resolved, unresolved } = resolveNames(names);
  assigneeIds = [...new Set([...assigneeIds, ...resolved])];
  if (unresolved.length > 0) {
    console.warn(`Warning: Could not resolve names: ${unresolved.join(', ')}`);
  }
}

async function main() {
  try {
    console.log('Creating task...');
    console.log(`  Title: ${values.title}`);
    console.log(`  Due: ${values.due}`);
    console.log(`  Timezone: ${values.timezone}`);
    console.log(`  Assignees: ${assigneeIds.map(id => getDisplayNameSync(id)).join(', ') || '(none)'}`);
    console.log('');
    
    const task = await createTask({
      title: values.title,
      description: values.description,
      dueTime: values.due,
      assigneeIds,
      timezone: values.timezone
    });
    
    console.log('✅ Task created successfully!');
    console.log('');
    console.log('Task Details:');
    console.log(`  Task ID: ${task.guid}`);
    console.log(`  Title: ${task.summary}`);
    console.log(`  URL: ${task.url || 'N/A'}`);
    
    // Output JSON for programmatic use
    if (process.env.JSON_OUTPUT) {
      console.log('\nJSON:');
      console.log(JSON.stringify(task, null, 2));
    }
    
  } catch (error) {
    console.error('❌ Failed to create task:', error.message);
    if (error.larkResponse) {
      console.error('Lark response:', JSON.stringify(error.larkResponse, null, 2));
    }
    process.exit(1);
  }
}

main();
