---
name: lark-calendar
description: Create, update, and delete calendar events and tasks in Lark (Feishu). Includes employee directory for automatic name-to-user_id resolution.
version: 1.0.0
author: Claw AI
---

# Lark Calendar & Task Skill

Create, update, and delete calendar events and tasks in Lark (Feishu).

## Overview

This skill provides full CRUD operations for:
- **Calendar Events** — meetings, appointments, schedules
- **Tasks (Todo)** — action items with deadlines

## Configuration

**Required Environment Variables** (in `.secrets.env`):
```bash
FEISHU_APP_ID=cli_a9f52a4ed7b8ded4
FEISHU_APP_SECRET=<your-app-secret>
```

**Default Calendar:** `feishu.cn_caF80RJxgGcbBGsQx64bCh@group.calendar.feishu.cn` (Claw calendar)

**Default Timezone:** `Asia/Singapore`

## Quick Reference

### Create Calendar Event

```bash
node skills/lark-calendar/scripts/create-event.mjs \
  --title "Meeting with Team" \
  --description "Discuss Q2 roadmap" \
  --start "2026-02-03 14:00:00" \
  --end "2026-02-03 15:00:00" \
  --attendees "Boyang,RK" \
  --location "Meeting Room A"
```

**Parameters:**
| Param | Required | Description |
|-------|----------|-------------|
| `--title` | ✅ | Event title |
| `--description` | ❌ | Event description |
| `--start` | ✅ | Start time (YYYY-MM-DD HH:MM:SS) |
| `--end` | ✅ | End time (YYYY-MM-DD HH:MM:SS) |
| `--attendees` | ❌ | Comma-separated names (auto-resolved to user_ids) |
| `--attendee-ids` | ❌ | Comma-separated user_ids directly |
| `--location` | ❌ | Event location |
| `--timezone` | ❌ | Timezone (default: Asia/Singapore) |
| `--calendar` | ❌ | Calendar ID (uses default if omitted) |

### Update Calendar Event

```bash
node skills/lark-calendar/scripts/update-event.mjs \
  --event-id "f9900f6b-b472-4b17-a818-7b5584abdc37_0" \
  --title "Updated Title" \
  --start "2026-02-03 15:00:00" \
  --end "2026-02-03 16:00:00"
```

### Delete Calendar Event

```bash
node skills/lark-calendar/scripts/delete-event.mjs \
  --event-id "f9900f6b-b472-4b17-a818-7b5584abdc37_0"
```

### List Calendar Events

```bash
# List events for next 7 days
node skills/lark-calendar/scripts/list-events.mjs

# List events in date range
node skills/lark-calendar/scripts/list-events.mjs \
  --start "2026-02-01" \
  --end "2026-02-28"
```

### Create Task

```bash
node skills/lark-calendar/scripts/create-task.mjs \
  --title "Review PR #123" \
  --description "Code review for authentication module" \
  --due "2026-02-05 18:00:00" \
  --assignees "Boyang,jc"
```

**Parameters:**
| Param | Required | Description |
|-------|----------|-------------|
| `--title` | ✅ | Task title |
| `--description` | ❌ | Task description |
| `--due` | ✅ | Due date (YYYY-MM-DD HH:MM:SS) |
| `--assignees` | ❌ | Comma-separated names (auto-resolved) |
| `--assignee-ids` | ❌ | Comma-separated user_ids directly |
| `--timezone` | ❌ | Timezone (default: Asia/Singapore) |

### Update Task

```bash
node skills/lark-calendar/scripts/update-task.mjs \
  --task-id "35fc5310-a1b1-49c7-be75-be631d3079ee" \
  --title "Updated Task" \
  --due "2026-02-06 18:00:00"
```

### Delete Task

```bash
node skills/lark-calendar/scripts/delete-task.mjs \
  --task-id "35fc5310-a1b1-49c7-be75-be631d3079ee"
```

### Manage Event Attendees

```bash
# Add attendees
node skills/lark-calendar/scripts/manage-attendees.mjs \
  --event-id "xxx" --add "RK,jc"

# Remove attendees  
node skills/lark-calendar/scripts/manage-attendees.mjs \
  --event-id "xxx" --remove "jc"
```

### Manage Task Members

```bash
# Add members
node skills/lark-calendar/scripts/manage-task-members.mjs \
  --task-id "xxx" --add "RK,jc"

# Remove members
node skills/lark-calendar/scripts/manage-task-members.mjs \
  --task-id "xxx" --remove "jc"
```

## Employee Directory

Names are auto-resolved to Lark user_ids. Supported names:

| user_id | Names | Role |
|---------|-------|------|
| `dgg163e1` | Boyang, by, 博洋 | Boss |
| `gb71g28b` | RK | Leadership, R&D |
| `53gc5724` | Ding | Leadership, Operations |
| `217ec2c2` | Charline | HR |
| `f2bfd283` | 曾晓玲, xiaoling | HR |
| `f26fe45d` | HH | Research |
| `45858f91` | zan, Eva | - |
| `7f79b6de` | Issac | Operations |
| `1fb2547g` | 王铁柱 | Operations |
| `e5997acd` | 尼克, Nico | Operations |
| `438c3c1f` | Ivan | Operations |
| `17g8bab2` | Dodo | R&D, Product |
| `73b45ec5` | 启超, QiChaoShi | R&D, Design |
| `d1978a39` | chenglin | R&D, Frontend |
| `ef6fc4a7` | 冠林, Green | R&D, Frontend |
| `b47fa8f2` | sixian, sx, Sixian-Yu | R&D, Frontend |
| `934fbf15` | jc, sagiri, 俊晨 | R&D, Backend |
| `8c4aad87` | 大明, daming | R&D, Backend |
| `ab87g5e1` | Emily Yobal | Intern |
| `55fa337f` | jingda, 景达 | Intern |
| `333c7cf1` | 刘纪源, 纪源, Aiden | Intern |

## Business Rules

1. **Boyang is always added** as attendee to every calendar event (automatic)
2. **Timezone handling:** Uses IANA identifiers (e.g., `Asia/Singapore`, `Asia/Shanghai`)
3. **Time format:** Always `YYYY-MM-DD HH:MM:SS`
4. **user_id vs open_id:** This skill uses `user_id` format (e.g., `dgg163e1`), NOT `open_id` (e.g., `ou_xxx`)

## Programmatic Usage

```javascript
import { createEvent, updateEvent, deleteEvent } from './skills/lark-calendar/lib/calendar.mjs';
import { createTask, updateTask, deleteTask } from './skills/lark-calendar/lib/task.mjs';
import { resolveNames } from './skills/lark-calendar/lib/employees.mjs';

// Create event
const result = await createEvent({
  title: 'Team Sync',
  description: 'Weekly standup',
  startTime: '2026-02-03 10:00:00',
  endTime: '2026-02-03 10:30:00',
  attendeeIds: ['dgg163e1', 'gb71g28b'],
  location: 'Zoom',
  timezone: 'Asia/Singapore'
});

// Create task
const task = await createTask({
  title: 'Review document',
  description: 'Q2 planning doc',
  dueTime: '2026-02-05 18:00:00',
  assigneeIds: ['dgg163e1'],
  timezone: 'Asia/Singapore'
});
```

## Lark API Reference

- [Calendar Events API](https://open.larksuite.com/document/server-docs/calendar-v4/calendar-event/create)
- [Calendar Attendees API](https://open.larksuite.com/document/server-docs/calendar-v4/calendar-event-attendee/create)
- [Tasks API](https://open.larksuite.com/document/server-docs/task-v2/task/create)

## Permissions Required

Ensure your Lark app has these scopes:
- `calendar:calendar` — Read/write calendar ✅ (already enabled)
- `calendar:calendar:readonly` — Read calendar ✅ (already enabled)
- `task:task:write` — Write tasks ⚠️ (needs to be added for task creation)
- `task:task:read` — Read tasks
- `contact:user.employee_id:readonly` — Read user info ✅ (already enabled)

**To add permissions:**
1. Go to [Lark Open Platform](https://open.larksuite.com/app/cli_a9f52a4ed7b8ded4/auth)
2. Add scopes: `task:task:write`, `contact:contact:readonly` (for dynamic employee lookup)
3. Re-publish the app version

**Note:** Without `contact:contact:readonly`, the skill uses a static fallback employee list. Update `lib/employees.mjs` when team changes.
