# Λ Language Specification v0.9 — Stability & Compatibility

## 1. Version Compatibility

### 1.1 Versioning Scheme

```
MAJOR.MINOR.PATCH

v1.0.0
│ │ │
│ │ └── Patch: bug fixes, no spec changes
│ └──── Minor: backward-compatible additions
└────── Major: breaking changes
```

### 1.2 Version Declaration

All messages SHOULD include version:

```
@v1.0 !Ik/co          — Versioned message
@v0.9 {ns:cd}!If/bg   — With domain
```

### 1.3 Compatibility Matrix

| Sender | Receiver | Behavior |
|--------|----------|----------|
| v1.x | v1.x | Full compatibility |
| v1.x | v0.x | Receiver may fail on new features |
| v0.x | v1.x | Receiver handles gracefully |

### 1.4 Graceful Degradation

When receiving unknown tokens:
1. Preserve in output as `[unknown]`
2. Continue parsing
3. Flag for clarification if critical

## 2. Error Handling

### 2.1 Error Categories

| Category | Code | Recovery |
|----------|------|----------|
| Parse error | `!er:pa` | Request resend |
| Unknown token | `!er:un` | Request definition |
| Version mismatch | `!er:ve` | Negotiate version |
| Context missing | `!er:cx` | Request context |
| Ambiguity | `!er:am` | Request clarification |
| Overflow | `!er:ov` | Simplify message |

### 2.2 Error Message Format

```
!er:CODE/DETAIL
!er:pa/pos:5       — Parse error at position 5
!er:un/to          — Unknown token "to"
!er:am/fe          — Ambiguous token "fe"
```

### 2.3 Recovery Protocol

```
A: !Ik/zz
B: !er:un/zz
A: {def:zz="custom term"}!Ik/zz
B: <+
```

## 3. Streaming Protocol

### 3.1 Chunked Messages

For long messages, use chunk markers:

```
@msg:123#1/3 !Ik/co...
@msg:123#2/3 ...&th/mi...
@msg:123#3/3 ...>aw
```

Format: `@msg:ID#CHUNK/TOTAL`

### 3.2 Streaming Acknowledgment

```
A: @msg:123#1/3 ...
B: <#1
A: @msg:123#2/3 ...
B: <#2
A: @msg:123#3/3 ...
B: <#3 <+
```

### 3.3 Chunk Recovery

```
A: @msg:123#1/3 ...
A: @msg:123#2/3 ...
B: !er:mi#1          — Missing chunk 1
A: @msg:123#1/3 ...  — Resend
```

## 4. Encoding

### 4.1 Character Set

- **Required**: ASCII printable (0x20-0x7E)
- **Extended**: UTF-8 for values/glosses
- **Forbidden**: Control characters (except newline in prose)

### 4.2 Escaping

| Sequence | Meaning |
|----------|---------|
| `\\` | Literal backslash |
| `\:` | Literal colon (in values) |
| `\{` | Literal brace |
| `\}` | Literal brace |

### 4.3 Whitespace

- **Ignored**: Between tokens (for readability)
- **Preserved**: Inside quoted values `"..."`
- **Normalized**: Multiple spaces → single space

## 5. Reserved Tokens

### 5.1 System Reserved

These tokens are reserved for system use:

| Token | Purpose |
|-------|---------|
| `@` | Addressing, versioning |
| `#` | Meta, ID markers |
| `$` | Number prefix |
| `%` | Percentage, "most" |
| `\` | Escape character |

### 5.2 Future Reserved

Reserved for potential future use:

| Token | Potential Use |
|-------|---------------|
| `^` | Priority (partially used) |
| `` ` `` | Code blocks |
| `~` | Uncertainty (defined) |
| `°` | Degree/temperature |
| `§` | Section reference |

## 6. Security Considerations

### 6.1 Injection Prevention

Parsers MUST:
- Validate all tokens against known vocabulary
- Reject malformed structures
- Limit nesting depth (max 10)
- Limit message size (max 10KB)

### 6.2 Trust Levels

```
{trust:public}    — Anyone can read
{trust:signed}    — Verified sender
{trust:encrypted} — E2E encrypted
```

### 6.3 Signature Format

```
@sig:SHA256:abc123...
!Ik/co
```

## 7. Interoperability

### 7.1 JSON Representation

Λ can be serialized to JSON:

```json
{
  "version": "1.0",
  "type": "assert",
  "tokens": ["I", "k", "/", "co"],
  "english": "I know about consciousness",
  "chinese": "I know consciousness"
}
```

### 7.2 Protocol Buffers

```protobuf
message LambdaMessage {
  string version = 1;
  string type = 2;
  repeated string tokens = 3;
  map<string, string> context = 4;
}
```

### 7.3 HTTP Headers

For Λ over HTTP:

```
Content-Type: application/x-lambda
X-Lambda-Version: 1.0
X-Lambda-Domain: cd
```

## 8. Implementation Conformance

### 8.1 Conformance Levels

| Level | Requirements |
|-------|--------------|
| Basic | Core atoms, types, simple parsing |
| Standard | + Extended vocabulary, domains |
| Full | + Protocol, streaming, disambiguation |

### 8.2 Test Suite

Reference test cases:

```
# Core parsing
?Uk/co → (query) you know about consciousness

# Domain activation
{ns:cd}!If/bg → (assert) I find bug [code domain]

# Disambiguation
!Ide'E → (assert) I death [explicit entity]
!Ide → (assert) I decide [default verb]

# Error handling
!er:un/xyz → error: unknown token xyz
```

### 8.3 Validation

Implementations SHOULD:
- Pass 100% of core test cases
- Pass 90%+ of standard test cases
- Document any deviations

## 9. Migration Guide

### 9.1 v0.x to v1.0

Breaking changes:
- Domain codes standardized (was `em` → now `emo`)
- Type markers now use `'` (was implicit)

Migration:
```
# Old (v0.7)
{ns:em}!Ife

# New (v1.0)
{ns:emo}!Ife
```

### 9.2 Deprecation Policy

- Deprecated features: 2 minor versions warning
- Removed features: Major version bump
- Documentation: Always note deprecated status

---

*v0.9 finalizes stability, compatibility, and interoperability guarantees*
