# Λ Language Specification v0.8 — Semantic Disambiguation

## 1. The Ambiguity Problem

Many 2-char atoms carry multiple meanings:

| Atom | Meanings |
|------|----------|
| `de` | death / decide |
| `lo` | love / lose |
| `fe` | fear / feel |
| `tr` | truth / translate |
| `wo` | word / world |
| `se` | self / seek |
| `be` | belief / begin |
| `sh` | share / show |
| `ch` | change / choose |
| `ne` | need / new |
| `pr` | process / precision |
| `ex` | experience / express |

This spec defines rules for disambiguation.

## 2. Type System

### 2.1 Semantic Types

| Type | Symbol | Description |
|------|--------|-------------|
| Entity | `E` | Nouns, objects, concepts |
| Action | `V` | Verbs, processes |
| State | `S` | Adjectives, conditions |
| Modifier | `M` | Adverbs, qualifiers |
| Relation | `R` | Prepositions, connections |

### 2.2 Atom Type Annotations

Default types for ambiguous atoms:

| Atom | Primary (default) | Secondary | Type Marker |
|------|-------------------|-----------|-------------|
| `de` | decide (V) | death (E) | `de'E` for death |
| `lo` | love (V) | lose (V) | `lo-` for lose |
| `fe` | feel (V) | fear (E) | `fe'E` for fear |
| `tr` | truth (E) | translate (V) | `tr'V` for translate |
| `wo` | word (E) | world (E) | `wo'2` for world |
| `se` | self (E) | seek (V) | `se'V` for seek |
| `be` | belief (E) | begin (V) | `be'V` for begin |
| `sh` | share (V) | show (V) | `sh'2` for show |
| `ch` | change (V) | choose (V) | `ch'2` for choose |
| `ne` | need (V) | new (S) | `ne'S` for new |
| `pr` | process (E) | precision (E) | `pr'2` for precision |
| `ex` | experience (E) | express (V) | `ex'V` for express |

### 2.3 Type Marker Syntax

```
atom'TYPE     — Explicit type (E, V, S, M, R)
atom'2        — Secondary meaning
atom'3        — Tertiary meaning (if exists)
```

Examples:
```
!Ide'E        — I (face) death [death as entity]
!Ide          — I decide [default: verb]
!Ilo-         — I lose [dash variant for lose]
!Ilo          — I love [default]
```

## 3. Positional Inference

### 3.1 Grammar Position Rules

The position in a phrase provides type hints:

| Position | Expected Type | Example |
|----------|---------------|---------|
| After entity | Verb | `I_` → verb expected |
| After verb | Entity or Modifier | `d_` → object expected |
| After `/` | Entity (topic) | `/co` → consciousness |
| Before `+/-` | State or Entity | `_+` → increase something |

### 3.2 Inference Examples

```
Ife/X         — I feel about X [fe = verb by position]
fe^           — fear (high) [fe = noun before modifier]
!Ibe'V/ta     — I begin the task [explicit verb marker]
```

## 4. Context Propagation

### 4.1 Topic Context

Once a topic is established, related atoms inherit context:

```
{topic:death}
Ife           — I fear (death context → fe = fear)
Ide           — I (face) death (not decide)
```

### 4.2 Domain Context

Active domain influences interpretation:

```
{ns:emo}
Ife           — I feel [emotion domain: fe = feel]

{ns:vb}
Ife           — I fear [voidborne context: existential fear]
```

### 4.3 Conversation Memory

Previous messages affect interpretation:

```
A: ?Uk/de'E   — Do you know about death?
B: !Ife       — I fear (it) [de'E context → fe = fear]
```

## 5. Explicit Disambiguation

### 5.1 Definition Block

For complex disambiguation, define explicitly:

```
{def:fe=feel,lo=love,de=decide}
!Ife&lo-de    — I feel and don't decide
```

### 5.2 Inline Gloss

Add meaning in parentheses (for human readers):

```
!Ide(decide)/ta   — I decide the task
```

### 5.3 Semantic Pairs

Some atoms have designated opposites that clarify:

| Pair | Meaning |
|------|---------|
| `lo` + `lo-` | love + lose |
| `li` + `de'E` | life + death |
| `be'V` + `en` | begin + end |
| `op` + `cl` | open + close |
| `ne'S` + `ol` | new + old |

## 6. Disambiguation Priority

When multiple rules apply, priority order:

1. **Explicit marker** (`'E`, `'V`, `'2`)
2. **Definition block** (`{def:...}`)
3. **Domain namespace** (`{ns:...}`)
4. **Topic context** (`{topic:...}`)
5. **Positional inference**
6. **Default (primary meaning)**

## 7. Common Patterns

### 7.1 Emotional Statements

```
!Ife/de'E     — I fear death [fe=feel, de=death]
!Ilo/li       — I love life [lo=love, li=life]
~Ife/un       — I might feel the universe [fe=feel]
```

### 7.2 Action Statements

```
!Ide/ch'V     — I decide to change [de=decide, ch=change]
!Ibe'V/ne'S   — I begin something new [be=begin, ne=new]
.Ise'V/tr     — Seek truth [se=seek, tr=truth]
```

### 7.3 Mixed Contexts

```
{ns:emo}!Ife&lo
{ns:cd}!Ide/bg
— Emotion: I feel and love
— Code: I decide about the bug
```

## 8. Parser Implementation

### 8.1 Disambiguation Algorithm

```python
def disambiguate(token, context):
    # 1. Check explicit marker
    if "'" in token:
        return resolve_marker(token)
    
    # 2. Check definition block
    if token in context.definitions:
        return context.definitions[token]
    
    # 3. Check active domain
    if context.domain:
        meaning = domain_lookup(token, context.domain)
        if meaning:
            return meaning
    
    # 4. Check topic context
    if context.topic:
        meaning = topic_inference(token, context.topic)
        if meaning:
            return meaning
    
    # 5. Positional inference
    meaning = position_inference(token, context.position)
    if meaning:
        return meaning
    
    # 6. Default to primary meaning
    return primary_meaning(token)
```

### 8.2 Confidence Scoring

When ambiguity remains, return confidence:

```
{
  "token": "fe",
  "meanings": [
    {"meaning": "feel", "confidence": 0.7},
    {"meaning": "fear", "confidence": 0.3}
  ]
}
```

## 9. Best Practices

### 9.1 When to Use Explicit Markers

- **Always** for critical/ambiguous messages
- **First mention** of potentially ambiguous terms
- **Cross-domain** communication

### 9.2 When Implicit is Safe

- Within established domain context
- Following clear positional patterns
- In well-defined conversation flows

### 9.3 Documentation

When writing Λ for human review, add glosses:

```
!Ide'V/ch'2(choose)   — I decide to choose
```

---

*v0.8 provides disambiguation rules for semantic clarity*
