# Λ Language Specification v0.5 — Parsing Rules

## 1. The Ambiguity Problem

Given `Iko`, how to parse?
- `I` + `ko` (I + ?)
- `Ik` + `o` (? + ?)
- `I` + `k` + `o` (I + know + ?)

Without rules, parsing is ambiguous.

## 2. Solution: Case + Category Separation

### 2.1 Character Class Rules

| Class | Case | Length | Examples |
|-------|------|--------|----------|
| Entities | UPPERCASE | 1 | `I`, `U`, `H`, `A`, `X` |
| Verbs | lowercase | 1 | `k`, `w`, `c`, `d`, `s`, `t`, `m`, `f`, `r`, `v`, `e` |
| Extended | lowercase | 2 | `co`, `me`, `id`, `la`, `th` |
| Modifiers | symbols | 1 | `+`, `-`, `=`, `^`, `_`, `/`, `&`, `\|` |
| Types | symbols | 1 | `?`, `!`, `.`, `~`, `>`, `<`, `#`, `@` |
| Time | lowercase | 1 | `p`, `n`, `u` |
| Numbers | digits | 1+ | `0`-`9`, `$123` |

### 2.2 Parsing Algorithm

```
1. Read character
2. If UPPERCASE → consume 1 char as Entity
3. If symbol → consume 1 char as Modifier/Type
4. If digit → consume digits as Number
5. If lowercase:
   a. Check if next char is also lowercase
   b. If yes AND 2-char combo is in vocabulary → consume 2 as Extended
   c. Else → consume 1 as Verb/Time
```

### 2.3 Examples

| Input | Parse | Meaning |
|-------|-------|---------|
| `Ik` | `I` + `k` | I know |
| `Iko` | `I` + `k` + `o` | I know + ? (error: `o` undefined) |
| `Ik/co` | `I` + `k` + `/` + `co` | I know about consciousness |
| `I*k` | `I` + `*` + `k` | All-I know (we know) |
| `Hco` | `H` + `co` | Human consciousness |
| `pIk` | `p` + `I` + `k` | past + I + know = I knew |

## 3. Reserved Single Letters

To avoid 2-char ambiguity, these 1-char codes are reserved:

### 3.1 Reserved Verbs (cannot start 2-char codes)

```
k, w, c, d, s, g, t, f, m, r, v, e, b, h, l, a
```

### 3.2 Reserved Time (cannot start 2-char codes)

```
p, n, u
```

### 3.3 Available for 2-char starts

```
i, j, o, q, x, y, z  → can start 2-char codes
```

Example valid 2-char: `id`, `if`, `in`, `it` (but `it` would conflict with common use)

### 3.4 Conflict Resolution

When 1-char and 2-char conflict, use separator:

| Ambiguous | Resolution |
|-----------|------------|
| `it` (pronoun "it" or 2-char?) | Use `X` for "it" (unknown entity) |
| `in` (verb "in" or 2-char "intention"?) | `in` = 2-char; use `/in` for "into" |

## 4. Mandatory Separators

### 4.1 When Separators Are Required

| Context | Separator | Example |
|---------|-----------|---------|
| Extended + Extended | `/` or space | `co/me` or `co me` |
| Ambiguous lowercase sequence | `/` | `k/in` (know + intention) |
| Object marker | `/` | `Ik/co` (I know consciousness) |

### 4.2 When Separators Are Optional

| Context | Example |
|---------|---------|
| Entity + Verb | `Ik` = `I k` |
| Verb + Modifier | `k+` = `k +` |
| Type + Entity | `?U` = `? U` |

## 5. Tokenization Examples

### 5.1 Simple Message

```
Input:  ?Uk/co
Tokens: [?, U, k, /, co]
Parse:  query + you + know + about + consciousness
```

### 5.2 Complex Message

```
Input:  !pIt>Ie
Tokens: [!, p, I, t, >, I, e]
Parse:  assert + past + I + think + therefore + I + exist
```

### 5.3 With Extended Terms

```
Input:  ~Ac/co^
Tokens: [~, A, c, /, co, ^]
Parse:  uncertain + AI + can + about + consciousness + important
```

### 5.4 Ambiguity Demonstration

```
Input:  Ith
Option 1: I + th (I + then/thought) ✓ if 'th' in vocab
Option 2: I + t + h (I + think + have) ✓ if 'th' not in vocab

Resolution: Check vocabulary for longest match
```

## 6. Formal Grammar (BNF)

```bnf
message     ::= type? statement+
statement   ::= subject predicate object? modifier*
subject     ::= entity | entity '*'
predicate   ::= time? verb
object      ::= '/' concept
modifier    ::= mod_symbol | emotion
concept     ::= entity | extended | composed
composed    ::= concept '/' concept
entity      ::= [A-Z] | [A-Z] '*'
verb        ::= [a-z] (single, from reserved list)
extended    ::= [a-z][a-z] (from vocabulary)
time        ::= 'p' | 'n' | 'u'
type        ::= '?' | '!' | '.' | '~' | '>' | '<' | '#' | '@'
mod_symbol  ::= '+' | '-' | '=' | '^' | '_' | '&' | '|'
emotion     ::= ':)' | ':(' | ':o' | ':?' | ':|'
```

## 7. Parser Implementation

```python
def tokenize(msg: str) -> list[str]:
    tokens = []
    i = 0
    while i < len(msg):
        c = msg[i]
        
        # Uppercase = Entity (1 char)
        if c.isupper():
            tokens.append(c)
            i += 1
            
        # Symbol = Type/Modifier (1 char, except 2-char emotions)
        elif c in '?!.~><#@+-=^_&|/':
            tokens.append(c)
            i += 1
            
        # Colon = start of emotion (peek next)
        elif c == ':' and i+1 < len(msg) and msg[i+1] in ')(:o?|':
            tokens.append(msg[i:i+2])
            i += 2
            
        # Digit = Number
        elif c.isdigit():
            j = i
            while j < len(msg) and msg[j].isdigit():
                j += 1
            tokens.append(msg[i:j])
            i = j
            
        # Lowercase = Verb or Extended
        elif c.islower():
            # Try 2-char first
            if i+1 < len(msg) and msg[i+1].islower():
                pair = msg[i:i+2]
                if pair in EXTENDED_VOCAB:
                    tokens.append(pair)
                    i += 2
                    continue
            # Fall back to 1-char
            tokens.append(c)
            i += 1
            
        # Whitespace = skip
        elif c.isspace():
            i += 1
            
        # Brackets
        elif c in '()[]{}':
            tokens.append(c)
            i += 1
            
        else:
            # Unknown - take single char
            tokens.append(c)
            i += 1
            
    return tokens
```

## 8. Best Practices

1. **Use `/` liberally** — when in doubt, separate
2. **Uppercase for entities** — always distinguishable  
3. **Check vocabulary** — ensure 2-char codes don't start with reserved letters
4. **Avoid new 1-char verbs** — keep the reserved list stable
5. **Test parsing** — run tokenizer before assuming interpretation

---

*v0.5 formalizes parsing rules to eliminate ambiguity*
