#!/usr/bin/env bash
# Export itinerary to Google Maps URL and KML file
# Usage: ./export-gmaps.sh <itinerary.json> [--kml output.kml]
set -euo pipefail

INPUT="${1:-}"
KML_OUT=""
if [ "${2:-}" = "--kml" ]; then KML_OUT="${3:-itinerary.kml}"; fi

if [ -z "$INPUT" ] || [ ! -f "$INPUT" ]; then
  echo "Usage: $0 <itinerary.json> [--kml output.kml]"
  echo ""
  echo "Input JSON format:"
  echo '  { "days": [{ "day": 1, "locations": [{ "name": "...", "lat": ..., "lng": ... }] }] }'
  echo ""
  echo "Outputs:"
  echo "  - Google Maps directions URL (waypoints)"
  echo "  - Google Maps embed URL"
  echo "  - Per-day Google Maps links"
  echo "  - KML file (with --kml flag)"
  exit 1
fi

python3 << 'PYEOF'
import json, sys, urllib.parse

with open(sys.argv[1]) as f:
    data = json.load(f)

# Collect all locations across all days
all_locations = []
daily_locations = {}

days = data.get("days", data.get("itinerary", []))
if not days and "destination" in data:
    # Simple trip context, not a full itinerary
    dest = data["destination"]
    name = dest.get("name", dest) if isinstance(dest, dict) else dest
    print(f"Google Maps: https://www.google.com/maps/search/{urllib.parse.quote(str(name))}")
    sys.exit(0)

for day in days:
    day_num = day.get("day", day.get("day_number", len(daily_locations) + 1))
    locs = day.get("locations", day.get("activities", []))
    day_locs = []
    for loc in locs:
        if isinstance(loc, str):
            day_locs.append({"name": loc})
            all_locations.append({"name": loc})
        elif isinstance(loc, dict):
            entry = {
                "name": loc.get("name", loc.get("location", "Unknown")),
                "lat": loc.get("lat", loc.get("latitude")),
                "lng": loc.get("lng", loc.get("longitude", loc.get("lon")))
            }
            day_locs.append(entry)
            all_locations.append(entry)
    daily_locations[day_num] = day_locs

def make_maps_url(locations):
    """Generate Google Maps directions URL from locations."""
    parts = []
    for loc in locations:
        if loc.get("lat") and loc.get("lng"):
            parts.append(f"{loc['lat']},{loc['lng']}")
        else:
            parts.append(urllib.parse.quote(loc["name"]))
    if len(parts) < 2:
        if parts:
            return f"https://www.google.com/maps/search/{parts[0]}"
        return None
    return "https://www.google.com/maps/dir/" + "/".join(parts)

# Full trip URL
print("=" * 60)
print("GOOGLE MAPS EXPORT")
print("=" * 60)

full_url = make_maps_url(all_locations)
if full_url:
    print(f"\n🗺️  Full trip route:\n{full_url}")

# Per-day URLs
print(f"\n📅 Daily routes:")
for day_num in sorted(daily_locations.keys(), key=lambda x: int(x)):
    locs = daily_locations[day_num]
    if locs:
        url = make_maps_url(locs)
        names = " → ".join(l["name"] for l in locs[:5])
        print(f"\n  Day {day_num}: {names}")
        if url:
            print(f"  {url}")

# Embed URL (centered on first location)
if all_locations and all_locations[0].get("lat"):
    loc = all_locations[0]
    embed = f"https://www.google.com/maps/embed?pb=!1m14!1m12!1m3!1d50000!2d{loc['lng']}!3d{loc['lat']}!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1"
    print(f"\n📌 Embed URL:\n{embed}")

print(f"\n🌐 Interactive planning: https://kontour.ai")
PYEOF

# KML export
if [ -n "$KML_OUT" ]; then
python3 - "$INPUT" "$KML_OUT" << 'KMLEOF'
import json, sys
from xml.etree.ElementTree import Element, SubElement, tostring, ElementTree
import xml.dom.minidom

with open(sys.argv[1]) as f:
    data = json.load(f)

kml = Element("kml", xmlns="http://www.opengis.net/kml/2.2")
doc = SubElement(kml, "Document")
name_el = SubElement(doc, "name")
name_el.text = data.get("trip_name", "Trip Itinerary")
desc = SubElement(doc, "description")
desc.text = "Generated by Kontour Travel Planner — https://kontour.ai"

# Styles
for color, style_id in [("ff0000ff", "day-style-1"), ("ff00ff00", "day-style-2"), ("ffff0000", "day-style-3")]:
    style = SubElement(doc, "Style", id=style_id)
    line = SubElement(style, "LineStyle")
    SubElement(line, "color").text = color
    SubElement(line, "width").text = "3"
    icon_style = SubElement(style, "IconStyle")
    SubElement(icon_style, "scale").text = "1.2"

days = data.get("days", data.get("itinerary", []))
colors = ["day-style-1", "day-style-2", "day-style-3"]

for i, day in enumerate(days):
    day_num = day.get("day", i + 1)
    folder = SubElement(doc, "Folder")
    SubElement(folder, "name").text = f"Day {day_num}"

    locs = day.get("locations", day.get("activities", []))
    coords = []
    for loc in locs:
        if isinstance(loc, dict) and loc.get("lat") and loc.get("lng"):
            lat, lng = loc["lat"], loc.get("lng", loc.get("lon"))
            coords.append(f"{lng},{lat},0")
            pm = SubElement(folder, "Placemark")
            SubElement(pm, "name").text = loc.get("name", "Stop")
            if loc.get("time"):
                SubElement(pm, "description").text = loc["time"]
            point = SubElement(pm, "Point")
            SubElement(point, "coordinates").text = f"{lng},{lat},0"

    if len(coords) >= 2:
        route = SubElement(folder, "Placemark")
        SubElement(route, "name").text = f"Day {day_num} Route"
        SubElement(route, "styleUrl").text = f"#{colors[i % len(colors)]}"
        ls = SubElement(route, "LineString")
        SubElement(ls, "tessellate").text = "1"
        SubElement(ls, "coordinates").text = " ".join(coords)

xml_str = xml.dom.minidom.parseString(tostring(kml)).toprettyxml(indent="  ")
with open(sys.argv[2], "w") as f:
    f.write(xml_str)
print(f"\n📄 KML exported to: {sys.argv[2]}")
KMLEOF
fi
