#!/usr/bin/env bash

set -euo pipefail

# Constants
VERSION="1.0.0"
CONFIG_DIR="$HOME/.config/kindroid"
CONFIG_FILE="$CONFIG_DIR/credentials.json"
RATE_LIMIT_FILE="/tmp/kindroid_last_call"
MIN_DELAY=3  # seconds between API calls
DEFAULT_TIMEOUT=300  # 5 minutes default timeout

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# Help text
usage() {
    cat << EOF
Kindroid CLI v${VERSION} 🤖

Usage: 
    $(basename $0) <command> [options] [arguments]

Commands:
    send    Send a message to your Kindroid
    break   Start a new conversation (chat break)
    status  Check companion status
    help    Show this help message

Options:
    -to <nickname>    Send to specific companion (default: uses default_ai)
    -t, --timeout N   Set API timeout in seconds (default: 300)
    -v, --verbose    Show detailed output

Examples:
    $(basename $0) send "Hello there!"
    $(basename $0) send -to alice "How are you?"
    $(basename $0) break "Let's start fresh"
EOF
    exit 1
}

# Error handling
error() {
    echo -e "${RED}Error: $1${NC}" >&2
    exit 1
}

warn() {
    echo -e "${YELLOW}Warning: $1${NC}" >&2
}

info() {
    echo -e "${GREEN}$1${NC}"
}

# Rate limiting
check_rate_limit() {
    if [[ -f "$RATE_LIMIT_FILE" ]]; then
        last_call=$(cat "$RATE_LIMIT_FILE")
        now=$(date +%s)
        elapsed=$((now - last_call))
        if (( elapsed < MIN_DELAY )); then
            sleep $((MIN_DELAY - elapsed))
        fi
    fi
    date +%s > "$RATE_LIMIT_FILE"
}

# Config validation
validate_config() {
    if [[ ! -f "$CONFIG_FILE" ]]; then
        error "Configuration not found at $CONFIG_FILE
Please create it with:
mkdir -p $CONFIG_DIR
echo '{
  \"default_ai\": \"your_ai_id\",
  \"api_key\": \"your_kn_key\",
  \"companions\": {}
}' > $CONFIG_FILE
chmod 600 $CONFIG_FILE"
    fi

    if [[ $(stat -c %a "$CONFIG_FILE") != "600" ]]; then
        warn "Fixing permissions on $CONFIG_FILE"
        chmod 600 "$CONFIG_FILE"
    fi
}

# Get credentials
get_credentials() {
    local companion="$1"
    validate_config
    
    API_KEY=$(jq -r '.api_key' "$CONFIG_FILE")
    if [[ "$companion" == "default" ]]; then
        AI_ID=$(jq -r '.default_ai' "$CONFIG_FILE")
    else
        AI_ID=$(jq -r ".companions.\"$companion\"" "$CONFIG_FILE")
    fi

    [[ "$API_KEY" == "null" ]] && error "API key not found in config"
    [[ "$AI_ID" == "null" ]] && error "AI ID not found for companion: $companion"
    [[ "$API_KEY" =~ ^kn_ ]] || error "Invalid API key format (should start with 'kn_')"
}

# API calls
api_call() {
    local endpoint="$1"
    local data="$2"
    local timeout="${3:-$DEFAULT_TIMEOUT}"

    check_rate_limit

    local response
    response=$(curl -s -X POST \
        "https://api.kindroid.ai/v1/$endpoint" \
        -H "Authorization: Bearer $API_KEY" \
        -H "Content-Type: application/json" \
        -d "$data" \
        --max-time "$timeout" || echo "ERROR: Connection failed")

    if [[ "$response" == ERROR* ]]; then
        error "Failed to connect to Kindroid API (timeout after ${timeout}s)"
    fi

    # Just echo the raw response - let the caller handle formatting
    echo "$response"
}

# Commands
cmd_send() {
    local companion="default"
    local timeout=$DEFAULT_TIMEOUT
    local message=""

    # Parse arguments
    while [[ $# -gt 0 ]]; do
        case $1 in
            -to)
                companion="$2"
                shift 2
                ;;
            -t|--timeout)
                timeout="$2"
                shift 2
                ;;
            *)
                message="$1"
                shift
                ;;
        esac
    done

    [[ -z "$message" ]] && error "Message required"

    get_credentials "$companion"
    
    info "Sending message to ${companion}..."
    echo "Waiting for response (timeout: ${timeout}s)..."
    api_call "send-message" "{\"ai_id\": \"$AI_ID\", \"message\": \"$message\"}" "$timeout"
}

cmd_break() {
    local companion="default"
    local greeting="$1"

    [[ -z "$greeting" ]] && error "Greeting message required for chat break"

    get_credentials "$companion"
    
    info "Starting new conversation..."
    api_call "chat-break" "{\"ai_id\": \"$AI_ID\", \"greeting\": \"$greeting\"}" >/dev/null
    info "Chat break successful"
}

cmd_status() {
    local companion="${1:-default}"
    get_credentials "$companion"
    
    info "Checking status for ${companion}..."
    api_call "status" "{\"ai_id\": \"$AI_ID\"}"
}

# Main
[[ $# -eq 0 ]] && usage

case "$1" in
    send)
        shift
        cmd_send "$@"
        ;;
    break)
        shift
        cmd_break "$@"
        ;;
    status)
        shift
        cmd_status "$@"
        ;;
    help)
        usage
        ;;
    *)
        error "Unknown command: $1"
        ;;
esac