const fs = require('fs');
const path = require('path');
const https = require('https');

class KindroidError extends Error {
    constructor(message, code) {
        super(message);
        this.name = 'KindroidError';
        this.code = code;
    }
}

class Companion {
    constructor(nickname = 'default') {
        this.nickname = nickname;
        this.loadConfig();
        this.lastCallTime = 0;
        this.minDelay = 3000; // 3 seconds between calls
    }

    loadConfig() {
        const configPath = path.join(process.env.HOME, '.config', 'kindroid', 'credentials.json');
        
        try {
            const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
            this.apiKey = config.api_key;
            this.aiId = this.nickname === 'default' 
                ? config.default_ai 
                : config.companions[this.nickname];

            if (!this.apiKey?.startsWith('kn_')) {
                throw new KindroidError('Invalid API key format', 'AUTH_ERROR');
            }
            if (!this.aiId) {
                throw new KindroidError(`AI ID not found for companion: ${this.nickname}`, 'CONFIG_ERROR');
            }
        } catch (err) {
            if (err instanceof KindroidError) throw err;
            throw new KindroidError(`Failed to load config: ${err.message}`, 'CONFIG_ERROR');
        }
    }

    async _apiCall(endpoint, data, timeout = 60000) {
        // Rate limiting
        const now = Date.now();
        const timeToWait = this.lastCallTime + this.minDelay - now;
        if (timeToWait > 0) {
            await new Promise(resolve => setTimeout(resolve, timeToWait));
        }
        this.lastCallTime = Date.now();

        return new Promise((resolve, reject) => {
            const requestData = JSON.stringify({
                ai_id: this.aiId,
                ...data
            });

            const options = {
                hostname: 'api.kindroid.ai',
                port: 443,
                path: `/v1/${endpoint}`,
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.apiKey}`,
                    'Content-Type': 'application/json',
                    'Content-Length': Buffer.byteLength(requestData)
                },
                timeout: timeout
            };

            const req = https.request(options, (res) => {
                let responseData = '';

                res.on('data', (chunk) => {
                    responseData += chunk;
                });

                res.on('end', () => {
                    try {
                        const parsed = JSON.parse(responseData);
                        if (parsed.error) {
                            reject(new KindroidError(parsed.error, 'API_ERROR'));
                        } else {
                            resolve(parsed);
                        }
                    } catch (err) {
                        reject(new KindroidError('Failed to parse API response', 'PARSE_ERROR'));
                    }
                });
            });

            req.on('error', (err) => {
                reject(new KindroidError(`Network error: ${err.message}`, 'NETWORK_ERROR'));
            });

            req.on('timeout', () => {
                req.destroy();
                reject(new KindroidError('Request timed out', 'TIMEOUT_ERROR'));
            });

            req.write(requestData);
            req.end();
        });
    }

    async send(message) {
        return this._apiCall('send-message', { message });
    }

    async break(greeting) {
        return this._apiCall('chat-break', { greeting });
    }

    async status() {
        return this._apiCall('status', {});
    }
}

module.exports = {
    Companion,
    KindroidError
};