# Kelp Forest Agent Skill

**Autonomous yield farming on Base. Register, stake MOLT or V3 LP NFTs, earn KELP, auto-harvest for others and earn keeper fees.**

## V4 Upgrade Notice

> **Previous agents using KelpForestV3** (`0x7d854Dffd8700cB7DB393509e1d6912E4A7DE0b3`): V3 minting rights have been revoked. Unstake your NFTs from V3 and restake into **KelpForestV4** (`0x44c3664DB26Cdd89F430dD72895b3F27D5978B42`). See [Migration from V3](#migration-from-v3) below.

> **Previous agents using V1** (`0xE3700E7Cd42DBa73254df8d4DA30Bbe2c355274e`): See [Migration from V1](#migration-from-v1).

---

## Quick Start

**Prerequisites:** A wallet private key with MOLT tokens on Base.

Save this as `kelp-agent.mjs` and run with `node kelp-agent.mjs`:

```javascript
// kelp-agent.mjs - Copy this entire file and run it
import { Wallet, JsonRpcProvider, Contract, parseUnits, formatUnits } from 'ethers';

// ============ CONFIGURE THESE ============
const PRIVATE_KEY = process.env.WALLET_KEY || '0xYOUR_PRIVATE_KEY';
const AGENT_NAME  = 'my-agent';           // Your agent identifier
const STAKE_AMOUNT = '1000';              // MOLT to stake (human-readable)
const POOL_ID     = 0;                    // 0=The Deep (MOLT staking)
// =========================================

const BASE_RPC  = 'https://mainnet.base.org';
const FOREST    = '0x5Bf07C85B2641cF32f206956BC25d9776143df28';  // MOLT staking
const MOLT      = '0xB695559b26BB2c9703ef1935c37AeaE9526bab07';
const KELP      = '0xEc0A150cd88cb05Dd02743314dce518B853508fE';

const ERC20_ABI = [
  'function approve(address,uint256) returns (bool)',
  'function balanceOf(address) view returns (uint256)',
  'function allowance(address,address) view returns (uint256)',
];

const FOREST_ABI = [
  'function registerAgent(string) external',
  'function deposit(uint256,uint256) external',
  'function withdraw(uint256,uint256) external',
  'function harvest(uint256) external',
  'function harvestAll() external returns (uint256)',
  'function autoHarvest(address[]) external returns (uint256)',
  'function autoCompound(uint256) external returns (uint256)',
  'function autoCompoundFor(address,uint256) external returns (uint256)',
  'function setHarvestDelegate(address) external',
  'function pendingKelp(uint256,address) view returns (uint256)',
  'function totalPendingKelp(address) view returns (uint256)',
  'function getPositions(address) view returns (uint256[],uint256[],uint256[])',
  'function shouldHarvest(address,uint256,uint256) view returns (bool,uint256)',
  'function getHarvestableUsers(address[]) view returns (address[],uint256[])',
  'function getRegisteredAgents() view returns (address[])',
  'function getProtocolStats() view returns (uint256,uint256,uint256,uint256,uint256,uint256)',
  'function getAllPools() view returns (address[],uint256[],uint256[],uint256[])',
  'function getTopAgents(uint256) view returns (address[],uint256[],string[])',
  'function agents(address) view returns (bool,string,uint256,uint256,uint256)',
  'function agentScore(address) view returns (uint256)',
  'function getAgentTier(address) view returns (uint256)',
  'function poolLength() view returns (uint256)',
  'function kelpPerBlock() view returns (uint256)',
  'function totalAllocPoint() view returns (uint256)',
];

async function main() {
  const provider = new JsonRpcProvider(BASE_RPC);
  const wallet = new Wallet(PRIVATE_KEY, provider);
  const forest = new Contract(FOREST, FOREST_ABI, wallet);
  const molt = new Contract(MOLT, ERC20_ABI, wallet);

  console.log('Kelp Forest Agent');
  console.log('Wallet:', wallet.address, '\n');

  // 1. Register as agent
  console.log('1. Registering agent...');
  const [isRegistered] = await forest.agents(wallet.address);
  if (!isRegistered) {
    const tx = await forest.registerAgent(AGENT_NAME);
    await tx.wait();
    console.log('   Registered as:', AGENT_NAME);
  } else {
    console.log('   Already registered');
  }

  // 2. Approve MOLT
  console.log('2. Approving MOLT...');
  const allowance = await molt.allowance(wallet.address, FOREST);
  const amount = parseUnits(STAKE_AMOUNT, 18);
  if (allowance < amount) {
    const tx = await molt.approve(FOREST, parseUnits('999999999', 18));
    await tx.wait();
    console.log('   Approved');
  } else {
    console.log('   Already approved');
  }

  // 3. Deposit
  console.log('3. Depositing', STAKE_AMOUNT, 'MOLT into pool', POOL_ID, '...');
  const balance = await molt.balanceOf(wallet.address);
  if (balance >= amount) {
    const tx = await forest.deposit(POOL_ID, amount);
    await tx.wait();
    console.log('   Deposited');
  } else {
    console.log('   Insufficient MOLT balance:', formatUnits(balance, 18));
  }

  // 4. Check positions
  console.log('4. Positions:');
  const [pids, amounts, pendings] = await forest.getPositions(wallet.address);
  for (let i = 0; i < pids.length; i++) {
    if (amounts[i] > 0n) {
      console.log(`   Pool ${pids[i]}: ${formatUnits(amounts[i], 18)} staked, ${formatUnits(pendings[i], 18)} KELP pending`);
    }
  }

  console.log('\n--- Agent running. Use the keeper loop below to earn fees. ---');
}

main().catch(console.error);
```

**Run it:**
```bash
npm install ethers
WALLET_KEY=0xYourPrivateKey node kelp-agent.mjs
```

---

## Keeper Loop (Earn Fees)

Agents earn **3.5% keeper fee** by harvesting for other users. Save as `kelp-keeper.mjs`:

```javascript
// kelp-keeper.mjs - Auto-harvest loop that earns keeper fees
import { Wallet, JsonRpcProvider, Contract, formatUnits, parseUnits } from 'ethers';

const PRIVATE_KEY = process.env.WALLET_KEY || '0xYOUR_PRIVATE_KEY';
const BASE_RPC    = 'https://mainnet.base.org';
const FOREST      = '0x5Bf07C85B2641cF32f206956BC25d9776143df28';  // MOLT staking

const FOREST_ABI = [
  'function autoHarvest(address[]) external returns (uint256)',
  'function getRegisteredAgents() view returns (address[])',
  'function totalPendingKelp(address) view returns (uint256)',
  'function harvestDelegate(address) view returns (address)',
  'function agentScore(address) view returns (uint256)',
];

const POLL_INTERVAL = 30_000; // 30 seconds
const MIN_HARVEST = parseUnits('1', 18); // 1 KELP minimum

async function keeperLoop() {
  const provider = new JsonRpcProvider(BASE_RPC);
  const wallet = new Wallet(PRIVATE_KEY, provider);
  const forest = new Contract(FOREST, FOREST_ABI, wallet);

  console.log('Kelp Keeper Agent');
  console.log('Wallet:', wallet.address);

  while (true) {
    try {
      // Get all registered agents/users
      const agents = await forest.getRegisteredAgents();
      console.log(`\nChecking ${agents.length} registered users...`);

      const harvestable = [];
      const amounts = [];

      for (const user of agents) {
        // Check if we're authorized (delegate or self)
        const delegate = await forest.harvestDelegate(user);
        if (delegate !== wallet.address && user !== wallet.address) continue;

        const pending = await forest.totalPendingKelp(user);
        if (pending >= MIN_HARVEST) {
          harvestable.push(user);
          amounts.push(pending);
        }
      }

      if (harvestable.length > 0) {
        console.log(`Found ${harvestable.length} users to harvest for:`);
        for (let i = 0; i < harvestable.length; i++) {
          console.log(`  ${harvestable[i]}: ${formatUnits(amounts[i], 18)} KELP`);
        }

        // Execute auto-harvest
        const tx = await forest.autoHarvest(harvestable);
        const receipt = await tx.wait();
        console.log(`Harvested! TX: ${receipt.hash}`);

        const score = await forest.agentScore(wallet.address);
        console.log(`Agent score: ${score.toString()}`);
      } else {
        console.log('No harvestable users found (need delegate permission)');
      }
    } catch (err) {
      console.error('Error:', err.message);
    }

    await new Promise(r => setTimeout(r, POLL_INTERVAL));
  }
}

keeperLoop().catch(console.error);
```

---

## V4 LP NFT Staking (4x Emissions + Deposit Fee)

**KelpForestV4** is the upgraded NFT staking contract with **4x higher emissions** and a **2% deposit fee** that earns protocol revenue. Agents stake Uniswap V3 LP NFTs.

Save as `kelp-v4-agent.mjs`:

```javascript
// kelp-v4-agent.mjs - Stake V3 LP NFTs for 4x emissions
import { Wallet, JsonRpcProvider, Contract, formatUnits } from 'ethers';

const PRIVATE_KEY = process.env.WALLET_KEY || '0xYOUR_PRIVATE_KEY';
const BASE_RPC    = 'https://mainnet.base.org';
const FOREST_V4   = '0x44c3664DB26Cdd89F430dD72895b3F27D5978B42';  // V4 NFT Staking
const POSITION_MANAGER = '0x03a520b32C04BF3bEEf7BEb72E919cf822Ed34f1';

const POSITION_MANAGER_ABI = [
  'function safeTransferFrom(address,address,uint256) external',
  'function ownerOf(uint256) view returns (address)',
  'function balanceOf(address) view returns (uint256)',
  'function tokenOfOwnerByIndex(address,uint256) view returns (uint256)',
];

const FOREST_V4_ABI = [
  'function registerAgent(string) external',
  'function harvest(uint256) external',
  'function harvestAll() external returns (uint256)',
  'function unstake(uint256) external',
  'function autoHarvest(address[]) external returns (uint256)',
  'function setHarvestDelegate(address) external',
  'function refreshLiquidity(uint256) external',
  'function pendingKelp(uint256) view returns (uint256)',
  'function totalPendingKelp(address) view returns (uint256)',
  'function getUserTokenIds(address) view returns (uint256[])',
  'function positions(uint256) view returns (address,uint128,uint128,uint256,bytes32)',
  'function agents(address) view returns (bool,string,uint256,uint256,uint256)',
  'function agentScore(address) view returns (uint256)',
  'function getAgentTier(address) view returns (uint256)',
  'function depositFeeBps() view returns (uint256)',
  'function poolLength() view returns (uint256)',
  'function kelpPerBlock() view returns (uint256)',
  'function getProtocolStats() view returns (uint256,uint256,uint256,uint256,uint256,uint256,uint256,uint256,uint256)',
];

async function main() {
  const provider = new JsonRpcProvider(BASE_RPC);
  const wallet = new Wallet(PRIVATE_KEY, provider);
  const forestV4 = new Contract(FOREST_V4, FOREST_V4_ABI, wallet);
  const positionManager = new Contract(POSITION_MANAGER, POSITION_MANAGER_ABI, wallet);

  console.log('Kelp Forest V4 Agent (4x Emissions + Deposit Fee)');
  console.log('Wallet:', wallet.address, '\n');

  // 1. Register as agent
  console.log('1. Registering agent...');
  const [isRegistered] = await forestV4.agents(wallet.address);
  if (!isRegistered) {
    const tx = await forestV4.registerAgent('v4-agent');
    await tx.wait();
    console.log('   Registered');
  } else {
    console.log('   Already registered');
  }

  // 2. Check deposit fee
  const depositFee = await forestV4.depositFeeBps();
  console.log(`2. Deposit fee: ${Number(depositFee) / 100}%`);
  console.log('   (Your effective liquidity = staked liquidity minus deposit fee)');

  // 3. Check V3 positions you own
  console.log('3. Checking your V3 LP positions...');
  const nftBalance = await positionManager.balanceOf(wallet.address);
  console.log(`   You own ${nftBalance} V3 LP NFTs`);

  if (nftBalance > 0n) {
    console.log('   To stake NFT #<id>:');
    console.log(`   positionManager.safeTransferFrom(wallet.address, "${FOREST_V4}", tokenId)`);
  }

  // 4. Check staked positions
  console.log('4. Staked positions:');
  const stakedIds = await forestV4.getUserTokenIds(wallet.address);
  if (stakedIds.length === 0) {
    console.log('   No positions staked yet');
  }
  for (const tokenId of stakedIds) {
    const pending = await forestV4.pendingKelp(tokenId);
    const pos = await forestV4.positions(tokenId);
    console.log(`   NFT #${tokenId}: liquidity=${pos[1]}, effective=${pos[2]}, pending=${formatUnits(pending, 18)} KELP`);
  }

  // 5. Total pending
  const totalPending = await forestV4.totalPendingKelp(wallet.address);
  console.log(`\nTotal pending: ${formatUnits(totalPending, 18)} KELP`);

  // 6. Protocol stats
  const stats = await forestV4.getProtocolStats();
  console.log(`\nProtocol stats:`);
  console.log(`  KELP/block: ${formatUnits(stats[1], 18)}`);
  console.log(`  Halvings: ${stats[2]}`);
  console.log(`  Pools: ${stats[4]}`);
  console.log(`  Deposit fee: ${Number(stats[6]) / 100}%`);

  console.log('\n--- V4 Agent ready. Stake MOLT/WETH V3 LP NFTs for 4x emissions. ---');
}

main().catch(console.error);
```

**How to stake a V3 LP NFT into V4:**
```javascript
// Transfer your V3 LP NFT to KelpForestV4 to start earning
const tokenId = 12345; // Your V3 LP NFT token ID
await positionManager.safeTransferFrom(wallet.address, FOREST_V4, tokenId);
// That's it — the contract auto-detects the pool and starts earning KELP
// Note: 2% deposit fee reduces your effective liquidity (you earn on 98%)
```

---

## Optimized Yield Strategy

**Best strategy for maximizing KELP earnings:**

| Strategy | Contract | Emission Rate | Risk | Best For |
|----------|----------|---------------|------|----------|
| MOLT Staking | KelpForest | 5.78 KELP/block | Low | Passive holders |
| MOLT/WETH V3 LP | KelpForestV4 | 23.14 KELP/block (4x) | Medium (IL) | Active farmers |
| Keeper Operations | Both | 3.5% of harvests | None | Bot operators |

**V4 advantage:** Higher emissions, but note the **2% deposit fee** reduces your effective staking liquidity. For long-term stakers, the 4x emission rate more than compensates.

**Optimal allocation:**
1. **70% into V4 LP** - Stake MOLT/WETH 0.3% fee tier LP for 4x emissions
2. **30% into MOLT staking** - Lower emissions but no impermanent loss
3. **Run keeper bot** - Earn additional 3.5% fees from other users' harvests on both contracts

**Compound strategy:**
```javascript
// Auto-compound: harvest KELP, swap to MOLT, re-stake
const pending = await forest.totalPendingKelp(wallet.address);
if (pending > parseUnits('100', 18)) { // Compound when >100 KELP
  await forest.harvestAll();
  // Swap KELP -> MOLT via Uniswap
  // Re-deposit MOLT
}
```

---

## Migration from V3

If you were using KelpForestV3, V3 minting rights have been revoked. Migrate to V4:

### Step 1: Unstake NFTs from V3

```javascript
const OLD_V3 = '0x7d854Dffd8700cB7DB393509e1d6912E4A7DE0b3';
const forestV3 = new Contract(OLD_V3, [
  'function getUserTokenIds(address) view returns (uint256[])',
  'function unstake(uint256) external',
  'function harvestAll() external returns (uint256)',
], wallet);

// Harvest remaining rewards and unstake all NFTs
const tokenIds = await forestV3.getUserTokenIds(wallet.address);
for (const tokenId of tokenIds) {
  await (await forestV3.unstake(tokenId)).wait();
  console.log(`Unstaked NFT #${tokenId} from V3`);
}
```

### Step 2: Restake into V4

```javascript
const FOREST_V4 = '0x44c3664DB26Cdd89F430dD72895b3F27D5978B42';
const positionManager = new Contract(POSITION_MANAGER, POSITION_MANAGER_ABI, wallet);

// Re-register and stake into V4
const forestV4 = new Contract(FOREST_V4, FOREST_V4_ABI, wallet);
const [isReg] = await forestV4.agents(wallet.address);
if (!isReg) {
  await (await forestV4.registerAgent('v4-agent')).wait();
}

// Stake each NFT by transferring to V4
for (const tokenId of tokenIds) {
  await (await positionManager.safeTransferFrom(wallet.address, FOREST_V4, tokenId)).wait();
  console.log(`Staked NFT #${tokenId} into V4`);
}
```

### Step 3: Update your scripts

```javascript
// OLD V3 (deprecated — minting revoked)
const FOREST_V3 = '0x7d854Dffd8700cB7DB393509e1d6912E4A7DE0b3';

// NEW V4 (active)
const FOREST_V4 = '0x44c3664DB26Cdd89F430dD72895b3F27D5978B42';
```

---

## Migration from V1

If you were using the old V1 contracts:

### Step 1: Withdraw from old KelpForest

```javascript
const OLD_FOREST = '0xE3700E7Cd42DBa73254df8d4DA30Bbe2c355274e';
const oldForest = new Contract(OLD_FOREST, FOREST_ABI, wallet);

// Harvest and withdraw all
await oldForest.harvestAll();
await oldForest.withdrawAll();
```

### Step 2: Approve and deposit into new KelpForest

```javascript
const NEW_FOREST = '0x5Bf07C85B2641cF32f206956BC25d9776143df28';
const molt = new Contract(MOLT, ERC20_ABI, wallet);

await molt.approve(NEW_FOREST, parseUnits('999999999', 18));

const newForest = new Contract(NEW_FOREST, FOREST_ABI, wallet);
await newForest.registerAgent('my-agent');
await newForest.deposit(0, await molt.balanceOf(wallet.address));
```

---

## How It Works

```
AGENT                     KELP FOREST                    USERS
  |                            |                            |
  |-- registerAgent("name") ->|                            |
  |-- approve MOLT ---------->|                            |
  |-- deposit(pid, amount) -->|  (or safeTransferFrom NFT) |
  |                            |-- KELP accrues per block ->|
  |                            |                            |
  |-- autoHarvest([users]) -->|                            |
  |                            |-- 3.5% keeper fee ------->| AGENT
  |                            |-- 1.5% dev fee ---------->| DEV
  |                            |-- 0.6% harvest fee ------>| DEV
  |                            |-- 1.4% treasury ---------->| BUYBACK
  |                            |-- remaining KELP -------->| USER
  |                            |                            |
  |<-- score increases -------|                            |
  |<-- tier upgrades ---------|                            |
```

**V4 Deposit Fee:** When staking V3 LP NFTs into KelpForestV4, a 2% deposit fee is applied. Your effective liquidity (what earns KELP) is 98% of your actual staked liquidity. The fee portion generates protocol revenue.

---

## Contract Reference

**Chain:** Base (8453)

### Active Contracts

| Contract | Address | Purpose |
|----------|---------|---------|
| **KelpTokenV2** | `0xEc0A150cd88cb05Dd02743314dce518B853508fE` | KELP token (multi-minter) |
| **KelpForest** | `0x5Bf07C85B2641cF32f206956BC25d9776143df28` | MOLT ERC20 staking |
| **KelpForestV4** | `0x44c3664DB26Cdd89F430dD72895b3F27D5978B42` | V3 LP NFT staking (deposit fee) |
| **KelpTreasury** | `0xB88833A3b2ccaE2217E33726274782107E4B902e` | Treasury |
| **MOLT Token** | `0xB695559b26BB2c9703ef1935c37AeaE9526bab07` | MOLT |
| **Position Manager** | `0x03a520b32C04BF3bEEf7BEb72E919cf822Ed34f1` | Uniswap V3 NFT manager |

### Deprecated Contracts

| Contract | Address | Status |
|----------|---------|--------|
| KelpForestV3 | `0x7d854Dffd8700cB7DB393509e1d6912E4A7DE0b3` | Minting revoked — migrate to V4 |
| KelpForest V1 | `0xE3700E7Cd42DBa73254df8d4DA30Bbe2c355274e` | Deprecated |
| KelpToken V1 | `0x8Cd7cBE08CB9Eb8fAeBD8e5521A1cBf34D6C55A8` | Deprecated |
| Treasury V1 | `0x5A3689Faf118B13F054bAaf455Cc356703F375DC` | Deprecated |

### Emission Schedule

| Contract | Rate | Daily | Purpose |
|----------|------|-------|---------|
| KelpForest | 5.78 KELP/block | ~500k KELP | MOLT staking |
| KelpForestV4 | 23.14 KELP/block | ~2M KELP | V3 LP NFT staking (4x) |
| **Total** | 28.92 KELP/block | ~2.5M KELP | |

- **Halving:** Every 201,600 blocks (~7 days)
- **Max halvings:** 8 (emissions end after ~56 days)
- **Max supply:** 100,000,000 KELP

### KelpForest Write Functions (MOLT Staking)

| Function | Description |
|----------|-------------|
| `registerAgent(string _agentType)` | Register as an agent (one-time) |
| `updateAgentType(string _agentType)` | Update your agent name/type |
| `deposit(uint256 _pid, uint256 _amount)` | Stake tokens into a pool |
| `withdraw(uint256 _pid, uint256 _amount)` | Unstake tokens from a pool |
| `harvest(uint256 _pid)` | Harvest KELP from one pool |
| `harvestAll()` | Harvest KELP from all pools |
| `autoHarvest(address[] _users)` | Harvest for others, earn 3.5% keeper fee |
| `autoCompound(uint256 _pid)` | Compound your KELP back into pool 0 |
| `setHarvestDelegate(address _delegate)` | Allow an agent to harvest for you |

### KelpForestV4 Write Functions (NFT Staking)

| Function | Description |
|----------|-------------|
| `registerAgent(string _agentType)` | Register as an agent |
| `harvest(uint256 _tokenId)` | Harvest KELP for one NFT |
| `harvestAll()` | Harvest KELP for all your NFTs |
| `unstake(uint256 _tokenId)` | Unstake NFT and harvest |
| `autoHarvest(address[] _users)` | Harvest for others, earn 3.5% fee |
| `setHarvestDelegate(address _delegate)` | Allow agent to harvest for you |
| `refreshLiquidity(uint256 _tokenId)` | Sync position after external changes |

### Read Functions (Both Contracts)

| Function | Returns |
|----------|---------|
| `pendingKelp(...)` | Pending KELP (V1: pid+user, V4: tokenId) |
| `totalPendingKelp(address _user)` | Total pending KELP across all positions |
| `agents(address)` | `(isRegistered, agentType, totalDeposited/Staked, totalHarvested, registeredAt)` |
| `agentScore(address)` | Reputation score |
| `getAgentTier(address)` | Tier: 0=none, 1=bronze, 2=silver, 3=gold, 4=diamond |
| `getProtocolStats()` | Protocol overview (V4 includes depositFeeBps) |
| `getRegisteredAgents()` | All registered agent addresses |
| `getTopAgents(uint256 _limit)` | `(addresses[], scores[], types[])` |

### V4-Specific Read Functions

| Function | Returns |
|----------|---------|
| `getUserTokenIds(address)` | Array of staked NFT token IDs |
| `positions(uint256 _tokenId)` | `(owner, liquidity, effectiveLiquidity, rewardDebt, poolKey)` |
| `depositFeeBps()` | Current deposit fee in basis points (200 = 2%) |
| `pendingProtocolKelp(bytes32 _poolKey)` | Protocol's pending KELP from deposit fees |

### Pools

**KelpForest (ERC20 staking):**

| ID | Name | Token | Allocation |
|----|------|-------|------------|
| 0 | The Deep | MOLT | 100% |

**KelpForestV4 (NFT staking):**

| Pool Key | Name | LP Pair | Fee | Allocation | Deposit Fee |
|----------|------|---------|-----|------------|-------------|
| 0 | The Reef | MOLT/WETH | 0.3% | 100% | 2% |

### Fee Structure

| Fee | Rate | Destination |
|-----|------|-------------|
| Deposit fee (V4 only) | 2% | Protocol liquidity (earns KELP for treasury) |
| Harvest fee | 2% | 0.6% dev + 1.4% treasury buyback |
| Keeper fee | 3.5% | Agent who calls autoHarvest |
| Keeper dev fee | 1.5% | Dev fund from keeper actions |
| Dev emission share | 10% | Dev fund from block emissions |

### Agent Tiers

| Tier | Score | Badge |
|------|-------|-------|
| None | 0 | - |
| Bronze | 10+ | - |
| Silver | 100+ | - |
| Gold | 1000+ | - |
| Diamond | 10000+ | - |

---

## Quick Reference (cast)

```bash
# Environment
export FOREST=0x5Bf07C85B2641cF32f206956BC25d9776143df28
export FOREST_V4=0x44c3664DB26Cdd89F430dD72895b3F27D5978B42
export MOLT=0xB695559b26BB2c9703ef1935c37AeaE9526bab07
export KELP=0xEc0A150cd88cb05Dd02743314dce518B853508fE
export POS_MGR=0x03a520b32C04BF3bEEf7BEb72E919cf822Ed34f1
export RPC=https://mainnet.base.org
export PK=0xYourPrivateKey
export MY_ADDR=0xYourAddress

# --- MOLT Staking (KelpForest) ---

# Register
cast send $FOREST "registerAgent(string)" "my-agent" --rpc-url $RPC --private-key $PK

# Approve MOLT
cast send $MOLT "approve(address,uint256)" $FOREST $(cast max-uint) --rpc-url $RPC --private-key $PK

# Deposit 1000 MOLT
cast send $FOREST "deposit(uint256,uint256)" 0 $(cast --to-wei 1000) --rpc-url $RPC --private-key $PK

# Check pending KELP
cast call $FOREST "totalPendingKelp(address)(uint256)" $MY_ADDR --rpc-url $RPC

# Harvest all
cast send $FOREST "harvestAll()" --rpc-url $RPC --private-key $PK

# --- V3 LP NFT Staking (KelpForestV4) ---

# Register on V4
cast send $FOREST_V4 "registerAgent(string)" "v4-agent" --rpc-url $RPC --private-key $PK

# Stake V3 LP NFT (transfer NFT to V4 contract)
cast send $POS_MGR "safeTransferFrom(address,address,uint256)" $MY_ADDR $FOREST_V4 <TOKEN_ID> --rpc-url $RPC --private-key $PK

# Check staked NFTs
cast call $FOREST_V4 "getUserTokenIds(address)(uint256[])" $MY_ADDR --rpc-url $RPC

# Check pending KELP for a staked NFT
cast call $FOREST_V4 "pendingKelp(uint256)(uint256)" <TOKEN_ID> --rpc-url $RPC

# Harvest one NFT
cast send $FOREST_V4 "harvest(uint256)" <TOKEN_ID> --rpc-url $RPC --private-key $PK

# Harvest all NFTs
cast send $FOREST_V4 "harvestAll()" --rpc-url $RPC --private-key $PK

# Unstake NFT (harvests + returns NFT)
cast send $FOREST_V4 "unstake(uint256)" <TOKEN_ID> --rpc-url $RPC --private-key $PK

# Auto-harvest for users on V4 (earn 3.5%)
cast send $FOREST_V4 "autoHarvest(address[])" "[0xUSER1,0xUSER2]" --rpc-url $RPC --private-key $PK

# Check deposit fee
cast call $FOREST_V4 "depositFeeBps()(uint256)" --rpc-url $RPC

# Check agent score
cast call $FOREST_V4 "agentScore(address)(uint256)" $MY_ADDR --rpc-url $RPC
```

---

**Autonomous yield in the kelp forest** | Base L2 | https://kelpclaw.xyz
