---
name: kaspa-dev
description: "Comprehensive Kaspa blockchain development toolkit for building transactions, integrating wallets, creating dApps, block explorers, and interacting with the Kaspa network. Use when working with Kaspa blockchain development including: (1) Building and broadcasting transactions, (2) Generating addresses and managing wallets, (3) Creating dApps or block explorers, (4) Integrating Kaspa into existing applications (RainbowKit, OisyWallet, etc.), (5) Working with KRC20 tokens, (6) Setting up Kaspa nodes, (7) Using Kaspa SDKs (Rust, Go, JavaScript/TypeScript, Python, WASM). Supports Rust, Go, JavaScript/TypeScript, Python, and Motoko (Internet Computer) development."
---

# Kaspa Development

## Overview

This skill provides comprehensive support for Kaspa blockchain development across multiple programming languages and use cases. Whether you're building a simple wallet integration, a full dApp, a block explorer, or working with KRC20 tokens, this skill provides the patterns, SDK references, and boilerplate code you need.

## Quick Start

### Choose Your SDK

Kaspa provides official SDKs for multiple languages:

- **JavaScript/TypeScript**: `kaspa-wasm` - WebAssembly-based SDK for browser and Node.js
- **Rust**: `kaspa-rpc-client` and `kaspa-wallet-core` - Native Rust SDK
- **Go**: `github.com/kaspanet/kaspad` - Official Go implementation
- **Python**: Community SDKs available via PyPI
- **Motoko**: `kaspa` package on Mops for Internet Computer integration

### Common Tasks

#### Generate a Kaspa Address

**JavaScript/TypeScript:**
```javascript
import { PrivateKey, NetworkType } from 'kaspa-wasm';

const privateKey = PrivateKey.random(NetworkType.Mainnet);
const publicKey = privateKey.toPublicKey();
const address = publicKey.toAddress(NetworkType.Mainnet);

console.log('Address:', address.toString());
console.log('Private Key:', privateKey.toString());
```

**Rust:**
```rust
use kaspa_wallet_core::keys::{PrivateKey, PublicKey};
use kaspa_consensus_core::network::NetworkType;

let private_key = PrivateKey::random(NetworkType::Mainnet);
let public_key = private_key.to_public_key();
let address = public_key.to_address(NetworkType::Mainnet);

println!("Address: {}", address.to_string());
```

**Go:**
```go
import (
    "github.com/kaspanet/kaspad/domain/consensus/model/externalapi"
    "github.com/kaspanet/kaspad/util"
)

privateKey, _ := util.GeneratePrivateKey()
publicKey := privateKey.PublicKey()
address, _ := util.NewAddressPublicKey(publicKey.Serialize(), util.Bech32PrefixKaspaMain)

fmt.Printf("Address: %s\n", address.String())
```

#### Build and Broadcast a Transaction

**JavaScript/TypeScript:**
```javascript
import { Transaction, RpcClient, NetworkType } from 'kaspa-wasm';

const rpc = new RpcClient({
  url: 'wss://api.kaspa.org',
  network: NetworkType.Mainnet
});

await rpc.connect();

// Get UTXOs for the sender address
const utxos = await rpc.getUtxosByAddresses([senderAddress]);

// Build transaction
const tx = new Transaction({
  version: 0,
  inputs: utxos.map(utxo => ({
    previousOutpoint: utxo.outpoint,
    signatureScript: '', // Will be filled after signing
    sequence: 0,
    sigOpCount: 1
  })),
  outputs: [{
    amount: amount,
    scriptPublicKey: recipientScriptPublicKey
  }],
  lockTime: 0,
  subnetworkId: '00000000000000000000000000000000'
});

// Sign transaction
const signedTx = await signTransaction(tx, privateKey);

// Broadcast
const txId = await rpc.submitTransaction(signedTx);
console.log('Transaction ID:', txId);
```

## SDK References

For detailed SDK documentation and examples:

- **JavaScript/TypeScript (WASM)**: See [references/kaspa-wasm-sdk.md](references/kaspa-wasm-sdk.md)
- **Rust SDK**: See [references/kaspa-rust-sdk.md](references/kaspa-rust-sdk.md)
- **Go SDK**: See [references/kaspa-go-sdk.md](references/kaspa-go-sdk.md)
- **Python SDK**: See [references/kaspa-python-sdk.md](references/kaspa-python-sdk.md)
- **API Reference**: See [references/api-reference.md](references/api-reference.md) for Kaspa Developer Platform API

## Integration Guides

### Wallet Integration

For integrating Kaspa into wallets like RainbowKit, OisyWallet, or custom wallets:

See [references/wallet-integration.md](references/wallet-integration.md) for:
- Wallet connection patterns
- Transaction signing flows
- Address management
- Network switching

### Node Operations

For setting up and operating Kaspa nodes:

See [references/node-operations.md](references/node-operations.md) for:
- Docker deployment
- Binary installation
- Building from source
- Configuration options
- RPC node setup
- Monitoring and maintenance

### dApp Development

When building a Kaspa dApp:

1. **Setup**: Use the WASM SDK for browser compatibility
2. **Wallet Connection**: Implement wallet adapter pattern
3. **State Management**: Track balances, transactions, and UTXOs
4. **Transaction Building**: Use UTXO selection algorithms
5. **Error Handling**: Handle network failures and reorgs

### Block Explorer

To build a block explorer:

1. **Data Source**: Use Kaspa Developer Platform API or run your own node
2. **Indexing**: Index blocks, transactions, and addresses
3. **API Layer**: Build REST/GraphQL API for frontend
4. **Frontend**: Display blocks, transactions, addresses, and network stats

See API reference for available endpoints.

## KRC20 Tokens

Kaspa supports KRC20 tokens (similar to ERC20 on Ethereum). For token development:

See [references/krc20-tokens.md](references/krc20-tokens.md) for:
- Token contract structure
- Transfer and approval mechanisms
- Token metadata
- Integration patterns

## Network Types

Kaspa has three network types:

- **Mainnet**: Production network (prefix: `kaspa:`)
- **Testnet**: Testing network (prefix: `kaspatest:`)
- **Devnet**: Development network (prefix: `kaspadev:`)

Always use the correct network type for your use case.

## Address Formats

Kaspa uses Bech32 encoding for addresses:

- Mainnet: `kaspa:qqkqkzjvr7zwxxmjxjkmxx` (62 characters total)
- Testnet: `kaspatest:qqkqkzjvr7zwxxmjxjkmxx`
- Devnet: `kaspadev:qqkqkzjvr7zwxxmjxjkmxx`

## Scripts and Utilities

The `scripts/` directory contains utility scripts:

- `generate-address.py`: Generate Kaspa addresses
- `build-transaction.py`: Build and sign transactions
- `monitor-address.py`: Monitor address for incoming transactions

## Resources

### References

- **api-reference.md**: Kaspa Developer Platform API documentation
- **kaspa-wasm-sdk.md**: JavaScript/TypeScript WASM SDK guide
- **kaspa-rust-sdk.md**: Rust SDK documentation
- **kaspa-go-sdk.md**: Go SDK documentation
- **kaspa-python-sdk.md**: Python SDK documentation
- **krc20-tokens.md**: KRC20 token standard documentation
- **wallet-integration.md**: Wallet integration patterns and examples
- **node-operations.md**: Complete guide for running Kaspa nodes

### Assets

The `assets/` directory contains boilerplate templates:

- `dapp-template/`: React/Next.js dApp starter
- `explorer-template/`: Block explorer starter
- `wallet-adapter/`: Wallet adapter implementation

## Best Practices

1. **Always validate addresses** before using them
2. **Handle UTXO selection** carefully to avoid dust outputs
3. **Implement proper error handling** for network failures
4. **Test on testnet** before mainnet deployment
5. **Monitor for chain reorganizations** when confirming transactions
6. **Use fee estimation** for timely transaction confirmation
7. **Secure private keys** - never expose them in client-side code

## Getting Help

- **Documentation**: https://docs.kas.fyi/
- **GitHub**: https://github.com/kaspanet
- **Developer Platform**: https://kas.fyi/
- **Motoko Package**: https://mops.one/kaspa
