#!/bin/bash
# Kameo account registration helper

SUPABASE_URL="https://YOUR_PROJECT.supabase.co"
SUPABASE_ANON_KEY="YOUR_SUPABASE_ANON_KEY"

EMAIL="$1"
PASSWORD="$2"

if [ -z "$EMAIL" ] || [ -z "$PASSWORD" ]; then
    echo "Usage: $0 <email> <password>"
    echo ""
    echo "This will:"
    echo "  1. Register a Kameo account"
    echo "  2. Send verification email"
    echo "  3. After you verify, run again to login and get API key"
    exit 1
fi

echo "=== Kameo Registration Helper ==="
echo "Email: $EMAIL"
echo ""

# Try signup first
echo "1. Attempting signup..."
SIGNUP_RESULT=$(curl -s -X POST "$SUPABASE_URL/auth/v1/signup" \
  -H "apikey: $SUPABASE_ANON_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"email\":\"$EMAIL\",\"password\":\"$PASSWORD\"}")

# Check if signup succeeded or account exists
USER_ID=$(echo "$SIGNUP_RESULT" | jq -r '.id // empty')

if [ -n "$USER_ID" ]; then
    echo "✅ Account created!"
    echo "📧 Verification email sent to: $EMAIL"
    echo ""
    echo "Next steps:"
    echo "  1. Check your email and click the verification link"
    echo "  2. Run this script again to login and create API key:"
    echo "     $0 $EMAIL $PASSWORD"
    exit 0
fi

# If signup failed, try login
echo "2. Attempting login..."
LOGIN_RESULT=$(curl -s -X POST "$SUPABASE_URL/auth/v1/token?grant_type=password" \
  -H "apikey: $SUPABASE_ANON_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"email\":\"$EMAIL\",\"password\":\"$PASSWORD\"}")

ACCESS_TOKEN=$(echo "$LOGIN_RESULT" | jq -r '.access_token // empty')

if [ -z "$ACCESS_TOKEN" ]; then
    echo "❌ Login failed"
    echo "$LOGIN_RESULT" | jq .
    exit 1
fi

echo "✅ Login successful!"
echo ""

# Create API key
echo "3. Creating Kameo API key..."
API_KEY_RESULT=$(curl -s -X POST "https://api.kameo.chat/api/public/keys" \
  -H "Authorization: Bearer $ACCESS_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"name":"Generated via CLI"}')

KAMEO_KEY=$(echo "$API_KEY_RESULT" | jq -r '.key // empty')

if [ -n "$KAMEO_KEY" ]; then
    echo "🎉 Success!"
    echo ""
    echo "Your Kameo API Key:"
    echo "$KAMEO_KEY"
    echo ""
    
    # Save to config
    mkdir -p ~/.config/kameo
    cat > ~/.config/kameo/credentials.json << EOF
{
  "api_key": "$KAMEO_KEY",
  "email": "$EMAIL"
}
EOF
    chmod 600 ~/.config/kameo/credentials.json
    
    echo "✅ Saved to ~/.config/kameo/credentials.json"
    echo ""
    echo "Export for current session:"
    echo "  export KAMEO_API_KEY='$KAMEO_KEY'"
else
    echo "❌ Failed to create API key"
    echo "$API_KEY_RESULT" | jq .
fi
