#!/bin/bash
# Kameo Video Generation Script
# Usage: ./generate_video.sh <image_path> <prompt> [output_file] [aspect_ratio]

set -e

# Configuration
API_KEY="${KAMEO_API_KEY:-}"
API_BASE="https://api.kameo.chat/api/public"

# Check for API key
if [ -z "$API_KEY" ]; then
    if [ -f ~/.config/kameo/credentials.json ]; then
        API_KEY=$(jq -r '.api_key' ~/.config/kameo/credentials.json)
    fi
fi

if [ -z "$API_KEY" ]; then
    echo "❌ Error: KAMEO_API_KEY not set"
    echo "Set it via: export KAMEO_API_KEY='kam_...'"
    echo "Or save to: ~/.config/kameo/credentials.json"
    exit 1
fi

# Parse arguments
IMAGE_PATH="$1"
PROMPT="$2"
OUTPUT_FILE="${3:-output.mp4}"
ASPECT_RATIO="${4:-9:16}"

if [ -z "$IMAGE_PATH" ] || [ -z "$PROMPT" ]; then
    echo "Usage: $0 <image_path> <prompt> [output_file] [aspect_ratio]"
    echo ""
    echo "Example:"
    echo "  $0 avatar.jpg \"Hello world\" video.mp4 9:16"
    echo ""
    echo "Aspect ratios: 9:16, 16:9, 1:1"
    exit 1
fi

if [ ! -f "$IMAGE_PATH" ]; then
    echo "❌ Error: Image file not found: $IMAGE_PATH"
    exit 1
fi

echo "🎬 Kameo Video Generation"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "📷 Image: $IMAGE_PATH"
echo "💬 Prompt: ${PROMPT:0:80}..."
echo "📐 Aspect: $ASPECT_RATIO"
echo ""

# Create request JSON
REQUEST_FILE=$(mktemp)
cat > "$REQUEST_FILE" << EOF
{
  "image_base64": "$(base64 -w0 "$IMAGE_PATH" 2>/dev/null || base64 -i "$IMAGE_PATH" | tr -d '\n')",
  "prompt": $(echo "$PROMPT" | jq -Rs .),
  "seconds": 5,
  "aspect_ratio": "$ASPECT_RATIO"
}
EOF

echo "🚀 Sending generation request..."

# Send request
RESPONSE=$(curl -s -X POST "$API_BASE/generate" \
  -H "X-API-Key: $API_KEY" \
  -H "Content-Type: application/json" \
  -d @"$REQUEST_FILE")

rm -f "$REQUEST_FILE"

# Parse response
STATUS=$(echo "$RESPONSE" | jq -r '.status // empty')
ERROR=$(echo "$RESPONSE" | jq -r '.error // empty')
VIDEO_URL=$(echo "$RESPONSE" | jq -r '.video_url // empty')
JOB_ID=$(echo "$RESPONSE" | jq -r '.job_id // empty')
PROCESSING_TIME=$(echo "$RESPONSE" | jq -r '.processing_time_ms // 0')

if [ "$STATUS" = "completed" ] && [ -n "$VIDEO_URL" ]; then
    PROCESSING_SEC=$(echo "scale=1; $PROCESSING_TIME / 1000" | bc)
    echo "✅ Generation completed in ${PROCESSING_SEC}s"
    echo ""
    echo "🎬 Video URL:"
    echo "$VIDEO_URL"
    echo ""
    echo "📋 Job ID: $JOB_ID"
    echo ""
    echo "💾 To download (may require browser):"
    echo "   $VIDEO_URL"
    echo ""
    echo "MEDIA: $VIDEO_URL"
else
    echo "❌ Generation failed"
    if [ -n "$ERROR" ] && [ "$ERROR" != "null" ]; then
        echo "Error: $ERROR"
    fi
    echo ""
    echo "Full response:"
    echo "$RESPONSE" | jq .
    exit 1
fi
