#!/bin/bash
# Kameo Prompt Enhancement - Use vision AI to create detailed scene descriptions

IMAGE_PATH="$1"
DIALOGUE="$2"

if [ -z "$IMAGE_PATH" ] || [ -z "$DIALOGUE" ]; then
    echo "Usage: $0 <image_path> <dialogue>"
    echo ""
    echo "Example:"
    echo "  $0 gakki.jpg \"こんにちは、私はガッキーです。愛してます。\""
    echo ""
    echo "This will:"
    echo "  1. Analyze the image using Gemini vision"
    echo "  2. Generate a detailed cinematic scene description"
    echo "  3. Embed your dialogue into the description"
    echo "  4. Output enhanced prompt ready for Kameo"
    exit 1
fi

if [ ! -f "$IMAGE_PATH" ]; then
    echo "❌ Image not found: $IMAGE_PATH"
    exit 1
fi

echo "🔍 Analyzing image with Gemini..."
echo ""

# Use Gemini to analyze image
ANALYSIS_PROMPT="Analyze this image in extreme detail for AI talking-head video generation. 

Provide a comprehensive scene description following this exact format:

[Detailed location/environment description with specific details], [complete person appearance: clothing colors/textures/style, accessories, hairstyle, facial features, expression, posture], [positioning in frame]. [Detailed lighting description]. The person looks directly into the camera with [specific expression details], speaking in [tone/manner], \"[DIALOGUE]\". The scene is captured in [specific shot type], framed at [camera angle], [framing details]. [Complete lighting description: source, quality, shadows].

Requirements:
- Be extremely detailed like a film cinematography script
- Describe textures, colors, materials specifically
- Detail exact camera work (shot type, angle, framing)
- Describe facial expression and eye contact precisely  
- Use present tense throughout
- Keep the dialogue placeholder as [DIALOGUE]
- Output ONLY the enhanced description, no preamble

Example style:
\"In a bright outdoor winter setting with soft, overcast daylight, a young woman with long dark hair wearing a white knitted winter hat with ear flaps and a colorful patterned sweater with a dark scarf stands centered in frame. Snow is visible on her nose and clothing. She looks directly into the camera with a warm, genuine smile, her eyes crinkling with joy, speaking in a cheerful, affectionate tone, '[DIALOGUE]'. The scene is captured in a medium close-up shot, framed at eye level. The lighting is natural and diffused, coming from the overcast sky above, creating soft, even illumination with no harsh shadows.\""

python3 << PYEOF
import base64
import json
import urllib.request
import os
import sys

api_key = os.environ.get('GOOGLE_API_KEY')
if not api_key:
    print("❌ GOOGLE_API_KEY not set", file=sys.stderr)
    sys.exit(1)

with open("$IMAGE_PATH", 'rb') as f:
    img_b64 = base64.b64encode(f.read()).decode()

payload = {
    "contents": [{
        "parts": [
            {"text": """$ANALYSIS_PROMPT"""},
            {
                "inline_data": {
                    "mime_type": "image/jpeg",
                    "data": img_b64
                }
            }
        ]
    }],
    "generationConfig": {
        "temperature": 0.4,
        "maxOutputTokens": 1024
    }
}

req = urllib.request.Request(
    f"https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash-exp:generateContent?key={api_key}",
    data=json.dumps(payload).encode(),
    headers={'Content-Type': 'application/json'}
)

try:
    with urllib.request.urlopen(req, timeout=30) as response:
        result = json.loads(response.read())
        text = result['candidates'][0]['content']['parts'][0]['text'].strip()
        
        # Replace [DIALOGUE] with actual dialogue
        enhanced = text.replace("[DIALOGUE]", "$DIALOGUE")
        
        print(enhanced)
except Exception as e:
    print(f"❌ Error: {e}", file=sys.stderr)
    sys.exit(1)
PYEOF

echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "✅ Enhanced prompt generated"
echo ""
echo "Use this with: scripts/generate_video.sh"
