#!/usr/bin/env python3
"""
Kalshi Markets CLI - Read-only market data and opportunity analysis.
No authentication required for public endpoints.
"""

import argparse
import json
import sys
from datetime import datetime, timezone
from typing import Optional
import requests

BASE_URL = "https://api.elections.kalshi.com/trade-api/v2"

def api_get(endpoint: str, params: Optional[dict] = None) -> dict:
    """Make unauthenticated GET request to Kalshi API."""
    url = f"{BASE_URL}{endpoint}"
    resp = requests.get(url, params=params, timeout=30)
    resp.raise_for_status()
    return resp.json()

def get_events(status: str = "open", limit: int = 20, cursor: str = None) -> dict:
    """Get list of events."""
    params = {"status": status, "limit": limit}
    if cursor:
        params["cursor"] = cursor
    return api_get("/events", params)

def get_markets(event_ticker: str = None, series_ticker: str = None, 
                status: str = "open", limit: int = 50, cursor: str = None) -> dict:
    """Get markets, optionally filtered by event or series."""
    params = {"status": status, "limit": limit}
    if event_ticker:
        params["event_ticker"] = event_ticker
    if series_ticker:
        params["series_ticker"] = series_ticker
    if cursor:
        params["cursor"] = cursor
    return api_get("/markets", params)

def get_market(ticker: str) -> dict:
    """Get single market by ticker."""
    return api_get(f"/markets/{ticker}")

def get_orderbook(ticker: str, depth: int = 10) -> dict:
    """Get orderbook for a market."""
    return api_get(f"/markets/{ticker}/orderbook", {"depth": depth})

def search_markets(query: str, limit: int = 20) -> list:
    """Search markets by text query (searches in titles)."""
    # Kalshi doesn't have a search endpoint, so we fetch and filter
    all_markets = []
    cursor = None
    query_lower = query.lower()
    
    while len(all_markets) < limit:
        result = get_markets(status="open", limit=100, cursor=cursor)
        markets = result.get("markets", [])
        
        for m in markets:
            title = (m.get("title") or "").lower()
            subtitle = (m.get("subtitle") or "").lower()
            ticker = (m.get("ticker") or "").lower()
            
            if query_lower in title or query_lower in subtitle or query_lower in ticker:
                all_markets.append(m)
                if len(all_markets) >= limit:
                    break
        
        cursor = result.get("cursor")
        if not cursor or not markets:
            break
    
    return all_markets[:limit]

def format_market(m: dict, verbose: bool = False) -> str:
    """Format market for display."""
    ticker = m.get("ticker", "?")
    title = m.get("title", "Unknown")
    yes_price = m.get("yes_ask", m.get("last_price", 0)) or 0
    no_price = 100 - yes_price if yes_price else 0
    volume = m.get("volume", 0) or 0
    
    # Calculate implied probability
    prob = yes_price / 100 if yes_price else 0.5
    
    line = f"{ticker}: {title}\n"
    line += f"  YES: {yes_price}¢ ({prob*100:.0f}%) | NO: {no_price}¢ | Vol: {volume:,}"
    
    if verbose:
        close_time = m.get("close_time")
        if close_time:
            line += f"\n  Closes: {close_time}"
        subtitle = m.get("subtitle")
        if subtitle:
            line += f"\n  {subtitle}"
    
    return line

def find_opportunities(min_certainty: float = 0.85, min_return: float = 0.10,
                       min_volume: int = 100) -> list:
    """
    Find high-certainty, high-payoff opportunities.
    
    Criteria:
    - Price implies ≥min_certainty probability (YES ≥85¢ or ≤15¢)
    - Expected return ≥min_return (10%)
    - Volume ≥min_volume for liquidity
    """
    opportunities = []
    cursor = None
    seen = set()
    
    while len(opportunities) < 50:
        result = get_markets(status="open", limit=100, cursor=cursor)
        markets = result.get("markets", [])
        
        for m in markets:
            ticker = m.get("ticker")
            if ticker in seen:
                continue
            seen.add(ticker)
            
            yes_price = m.get("yes_ask") or m.get("last_price") or 50
            volume = m.get("volume") or 0
            
            if volume < min_volume:
                continue
            
            # Check for high-certainty YES (price ≥ 85¢)
            if yes_price >= min_certainty * 100:
                prob = yes_price / 100
                cost = yes_price  # cents to buy YES
                payoff = 100 - cost  # profit if YES wins
                expected_return = (prob * payoff - (1 - prob) * cost) / cost
                
                if expected_return >= min_return:
                    opportunities.append({
                        "market": m,
                        "side": "YES",
                        "price": yes_price,
                        "implied_prob": prob,
                        "potential_return": payoff / cost,
                        "expected_return": expected_return,
                        "volume": volume
                    })
            
            # Check for high-certainty NO (YES price ≤ 15¢)
            elif yes_price <= (1 - min_certainty) * 100:
                prob = 1 - (yes_price / 100)  # NO probability
                cost = 100 - yes_price  # cents to buy NO
                payoff = 100 - cost  # profit if NO wins
                expected_return = (prob * payoff - (1 - prob) * cost) / cost
                
                if expected_return >= min_return:
                    opportunities.append({
                        "market": m,
                        "side": "NO",
                        "price": 100 - yes_price,
                        "implied_prob": prob,
                        "potential_return": payoff / cost,
                        "expected_return": expected_return,
                        "volume": volume
                    })
        
        cursor = result.get("cursor")
        if not cursor or not markets:
            break
    
    # Sort by expected return
    opportunities.sort(key=lambda x: x["expected_return"], reverse=True)
    return opportunities

def format_opportunity(opp: dict) -> str:
    """Format opportunity for display."""
    m = opp["market"]
    ticker = m.get("ticker", "?")
    title = m.get("title", "Unknown")
    
    line = f"📈 {ticker}: {title}\n"
    line += f"   Side: {opp['side']} @ {opp['price']:.0f}¢\n"
    line += f"   Implied Prob: {opp['implied_prob']*100:.1f}%\n"
    line += f"   Potential Return: {opp['potential_return']*100:.1f}%\n"
    line += f"   Expected Return: {opp['expected_return']*100:.1f}%\n"
    line += f"   Volume: {opp['volume']:,}"
    return line

def cmd_trending(args):
    """Show trending/active markets."""
    result = get_markets(status="open", limit=args.limit)
    markets = result.get("markets", [])
    
    # Sort by volume
    markets.sort(key=lambda x: x.get("volume") or 0, reverse=True)
    
    print(f"📊 Top {len(markets)} Markets by Volume\n")
    for m in markets[:args.limit]:
        print(format_market(m, verbose=args.verbose))
        print()

def cmd_search(args):
    """Search markets by query."""
    markets = search_markets(args.query, limit=args.limit)
    
    if not markets:
        print(f"No markets found matching '{args.query}'")
        return
    
    print(f"🔍 Found {len(markets)} markets matching '{args.query}'\n")
    for m in markets:
        print(format_market(m, verbose=args.verbose))
        print()

def cmd_market(args):
    """Get details for a specific market."""
    try:
        result = get_market(args.ticker)
        m = result.get("market", result)
        
        print(format_market(m, verbose=True))
        print()
        
        # Get orderbook
        ob = get_orderbook(args.ticker, depth=5)
        print("📖 Orderbook:")
        
        bids = ob.get("orderbook", {}).get("yes", [])
        if bids:
            print("  YES bids:", ", ".join(f"{b[0]}¢ x{b[1]}" for b in bids[:5]))
        
        no_bids = ob.get("orderbook", {}).get("no", [])
        if no_bids:
            print("  NO bids:", ", ".join(f"{b[0]}¢ x{b[1]}" for b in no_bids[:5]))
            
    except requests.HTTPError as e:
        print(f"Error fetching market {args.ticker}: {e}")
        sys.exit(1)

def cmd_opportunities(args):
    """Find high-certainty, high-payoff opportunities."""
    print("🔎 Scanning for opportunities...\n")
    
    opps = find_opportunities(
        min_certainty=args.certainty,
        min_return=args.return_threshold,
        min_volume=args.min_volume
    )
    
    if not opps:
        print("No opportunities found matching criteria.")
        print(f"  Certainty: ≥{args.certainty*100:.0f}%")
        print(f"  Expected Return: ≥{args.return_threshold*100:.0f}%")
        print(f"  Min Volume: {args.min_volume}")
        return
    
    print(f"Found {len(opps)} opportunities:\n")
    for opp in opps[:args.limit]:
        print(format_opportunity(opp))
        print()

def cmd_events(args):
    """List events."""
    result = get_events(status="open", limit=args.limit)
    events = result.get("events", [])
    
    print(f"📅 {len(events)} Open Events\n")
    for e in events:
        ticker = e.get("event_ticker", "?")
        title = e.get("title", "Unknown")
        category = e.get("category", "")
        print(f"{ticker}: {title}")
        if category:
            print(f"  Category: {category}")
        print()

def main():
    parser = argparse.ArgumentParser(description="Kalshi Markets CLI")
    parser.add_argument("-v", "--verbose", action="store_true", help="Verbose output")
    
    subparsers = parser.add_subparsers(dest="command", required=True)
    
    # trending
    p_trending = subparsers.add_parser("trending", help="Show trending markets")
    p_trending.add_argument("-n", "--limit", type=int, default=10)
    p_trending.set_defaults(func=cmd_trending)
    
    # search
    p_search = subparsers.add_parser("search", help="Search markets")
    p_search.add_argument("query", help="Search query")
    p_search.add_argument("-n", "--limit", type=int, default=10)
    p_search.set_defaults(func=cmd_search)
    
    # market
    p_market = subparsers.add_parser("market", help="Get market details")
    p_market.add_argument("ticker", help="Market ticker")
    p_market.set_defaults(func=cmd_market)
    
    # opportunities
    p_opps = subparsers.add_parser("opportunities", help="Find high-value opportunities")
    p_opps.add_argument("-c", "--certainty", type=float, default=0.85,
                        help="Min implied probability (default: 0.85)")
    p_opps.add_argument("-r", "--return-threshold", type=float, default=0.10,
                        help="Min expected return (default: 0.10)")
    p_opps.add_argument("-m", "--min-volume", type=int, default=100,
                        help="Min volume (default: 100)")
    p_opps.add_argument("-n", "--limit", type=int, default=20)
    p_opps.set_defaults(func=cmd_opportunities)
    
    # events
    p_events = subparsers.add_parser("events", help="List events")
    p_events.add_argument("-n", "--limit", type=int, default=20)
    p_events.set_defaults(func=cmd_events)
    
    args = parser.parse_args()
    args.func(args)

if __name__ == "__main__":
    main()
