#!/usr/bin/env bash
# Execute a Jupyter notebook on Kaggle's Kernel Backend (KKB) using kaggle-cli.
#
# Workflow:
#   1. kaggle kernels push   — upload notebook and trigger execution
#   2. kaggle kernels status  — poll for completion
#   3. kaggle kernels output  — download execution results
#
# Prerequisites:
#   pip install kaggle
#   Credentials configured in ~/.kaggle/kaggle.json
#   kernel-metadata.json in the same directory as the notebook
#
# Usage:
#   bash scripts/cli_execute.sh <notebook-dir> <kernel-slug> [output-dir]
#
# Arguments:
#   notebook-dir  — directory containing notebook + kernel-metadata.json
#   kernel-slug   — e.g., "username/kernel-name"
#   output-dir    — directory to save output (default: ./downloads/notebook-output)

set -euo pipefail

NOTEBOOK_DIR="${1:?Usage: cli_execute.sh <notebook-dir> <kernel-slug> [output-dir]}"
KERNEL_SLUG="${2:?Usage: cli_execute.sh <notebook-dir> <kernel-slug> [output-dir]}"
OUTPUT_DIR="${3:-./downloads/notebook-output}"

echo "============================================================"
echo "Step 0: Prepare kernel-metadata.json"
echo "============================================================"
echo ""
echo "Before pushing, edit kernel-metadata.json and replace YOUR_USERNAME"
echo "with your actual Kaggle username. The metadata file must be in the"
echo "same directory as the notebook."
echo ""
echo "Required fields in kernel-metadata.json:"
echo '  "id": "username/kernel-slug"'
echo '  "code_file": "sample_notebook.ipynb"'
echo '  "language": "python"'
echo '  "kernel_type": "notebook"'
echo ""

echo "============================================================"
echo "Step 1: Push notebook to Kaggle (triggers execution)"
echo "============================================================"

# Push the notebook — this uploads the code and starts execution on KKB
kaggle kernels push -p "${NOTEBOOK_DIR}"

echo "Notebook pushed. Execution started on KKB."
echo ""

echo "============================================================"
echo "Step 2: Poll for execution status"
echo "============================================================"

# Poll every 30 seconds until the kernel completes
while true; do
    STATUS=$(kaggle kernels status "${KERNEL_SLUG}" 2>&1)
    echo "Status: ${STATUS}"

    if echo "${STATUS}" | grep -q "complete"; then
        echo "Execution complete!"
        break
    elif echo "${STATUS}" | grep -q "error"; then
        echo "Execution failed!"
        exit 1
    fi

    echo "Still running... waiting 30 seconds"
    sleep 30
done

echo ""
echo "============================================================"
echo "Step 3: Download execution output"
echo "============================================================"

# Download the output files generated by the notebook
mkdir -p "${OUTPUT_DIR}"
kaggle kernels output "${KERNEL_SLUG}" \
    --path "${OUTPUT_DIR}"

echo "Output downloaded to ${OUTPUT_DIR}/"
ls -la "${OUTPUT_DIR}/"

echo ""
echo "============================================================"
echo "Optional: Pull the executed notebook (with cell outputs)"
echo "============================================================"

# Pull the notebook source (includes rendered outputs after execution)
kaggle kernels pull "${KERNEL_SLUG}" \
    --path "${OUTPUT_DIR}/notebook-source" \
    --metadata

echo "Notebook source pulled to ${OUTPUT_DIR}/notebook-source/"
ls -la "${OUTPUT_DIR}/notebook-source/"
