"""Phase 5: Streak badges (~4 badges).

Generates a daily streak script and prints manual setup instructions:
  - 7-Day Login Streak
  - 30-Day Login Streak
  - Submission Streak (7 days)
  - Super Submission Streak (30 days)

The script makes a Kaggle API call (counts as login) and submits to the
Titanic competition. Users must manually configure cron/launchd to run it.
"""

import platform
import stat
from pathlib import Path

from badge_tracker import set_status, should_attempt
from utils import REPO_ROOT, TEMPLATES_DIR, get_kaggle_cli, get_username


DAILY_SCRIPT_PATH = REPO_ROOT / "skills" / "badge-collector" / "scripts" / "daily_streak.sh"


def _create_daily_script() -> Path:
    """Create the daily streak script."""
    kaggle_cli = get_kaggle_cli()
    submission_file = TEMPLATES_DIR / "submission_titanic.csv"

    script = f"""#!/usr/bin/env bash
# Daily Kaggle streak script — auto-generated by badge-collector
# Makes an API call (counts as login) and submits to Titanic

set -euo pipefail

KAGGLE="{kaggle_cli}"

# 1. API activity = login
$KAGGLE datasets list --page-size 1 > /dev/null 2>&1 || true

# 2. Daily Titanic submission (if submission file exists)
SUBMISSION="{submission_file}"
if [ -f "$SUBMISSION" ]; then
    $KAGGLE competitions submit \\
        -c titanic \\
        -f "$SUBMISSION" \\
        -m "Daily streak submission $(date +%Y-%m-%d)" \\
        > /dev/null 2>&1 || true
fi

echo "[$(date)] Streak script completed"
"""
    DAILY_SCRIPT_PATH.write_text(script)
    DAILY_SCRIPT_PATH.chmod(DAILY_SCRIPT_PATH.stat().st_mode | stat.S_IEXEC)
    return DAILY_SCRIPT_PATH


def _print_setup_instructions(script_path: Path) -> None:
    """Print platform-specific manual setup instructions."""
    system = platform.system()

    print("\n  To earn streak badges, run this script daily for 7/30 days.")
    print(f"  Script location: {script_path}")
    print()

    if system == "Darwin":
        print("  --- macOS: manual launchd setup ---")
        print(f"  1. Create a plist at ~/Library/LaunchAgents/com.kaggle.streak.plist")
        print(f"     with ProgramArguments pointing to: /bin/bash {script_path}")
        print(f"     and a StartCalendarInterval for your preferred time (e.g. 10:00 AM).")
        print(f"  2. Load it: launchctl load ~/Library/LaunchAgents/com.kaggle.streak.plist")
        print(f"  3. To stop: launchctl unload ~/Library/LaunchAgents/com.kaggle.streak.plist")
    else:
        print("  --- Linux: manual cron setup ---")
        print(f"  1. Open crontab: crontab -e")
        print(f"  2. Add this line (runs daily at 10:00 AM):")
        print(f"     0 10 * * * {script_path}")
        print(f"  3. Save and exit. Verify with: crontab -l")

    print()
    print(f"  Or run manually anytime: bash {script_path}")


def run(username: str) -> tuple[int, int]:
    """Generate daily streak script and print setup instructions.

    Returns (attempted, succeeded).
    """
    badge_ids = [
        "seven_day_login_streak",
        "thirty_day_login_streak",
        "submission_streak",
        "super_submission_streak",
    ]

    actionable = [b for b in badge_ids if should_attempt(b)]
    if not actionable:
        print("  All streak badges already earned/skipped")
        return 0, 0

    print("\n  --- Phase 5: Streak badge setup ---")

    # Create the daily script
    script_path = _create_daily_script()
    print(f"  Created daily script: {script_path}")

    # Print manual setup instructions (no auto-install)
    _print_setup_instructions(script_path)

    for bid in actionable:
        set_status(bid, "attempting", "daily script created — run manually or schedule via cron/launchd")

    print("\n  Streak setup complete!")
    print("  Badges will be earned after 7/30 days of daily execution.")
    return len(actionable), len(actionable)
