#!/bin/bash
# Install automatic skill update feedback loop in a project
# This is a meta-enforcement: ensures enforcement improvements flow back to skills

set -e

if [ -z "$1" ]; then
    echo "Usage: $0 /path/to/project [skill-name]"
    echo ""
    echo "Example: $0 /home/user/my-project agent-guardrails"
    exit 1
fi

PROJECT_DIR="$1"
SKILL_NAME="${2:-agent-guardrails}"

if [ ! -d "$PROJECT_DIR" ]; then
    echo "Error: $PROJECT_DIR does not exist"
    exit 1
fi

echo "🔄 Installing skill update feedback loop in $PROJECT_DIR"
echo "   Target skill: $SKILL_NAME"
echo ""

# Determine skill directory
if [ -d "$PROJECT_DIR/skills/$SKILL_NAME" ]; then
    SKILLS_DIR="$PROJECT_DIR/skills"
elif [ -d "$PROJECT_DIR/../skills/$SKILL_NAME" ]; then
    SKILLS_DIR="$PROJECT_DIR/../skills"
else
    echo "⚠️  Warning: Couldn't find skills/$SKILL_NAME"
    echo "   Will create generic configuration"
fi

# 1. Create detection script
cat > "$PROJECT_DIR/scripts/detect-enforcement-improvement.sh" << 'DETECTION_EOF'
#!/bin/bash
# Auto-detect enforcement improvements and create skill update tasks
# Generated by agent-guardrails/scripts/install-skill-feedback-loop.sh

WORKSPACE=$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)

# Patterns that indicate enforcement improvements
ENFORCEMENT_PATTERNS=(
    "\.deployment-check\.sh"
    "DEPLOYMENT-CHECKLIST\.md"
    "pre-commit"
    "post-create"
    "check-secrets"
    "\.guardrails"
    "enforcement"
    "mechanical"
)

# Check last commit
LAST_COMMIT=$(git log -1 --name-only --pretty=format:"")

echo "🔍 Checking for enforcement improvements in last commit..."

FOUND=()
for file in $LAST_COMMIT; do
    for pattern in "${ENFORCEMENT_PATTERNS[@]}"; do
        if echo "$file" | grep -q "$pattern"; then
            FOUND+=("$file")
            break
        fi
    done
done

if [ ${#FOUND[@]} -eq 0 ]; then
    echo "   No enforcement improvements detected"
    exit 0
fi

echo ""
echo "🎯 Detected ${#FOUND[@]} enforcement improvements:"
for file in "${FOUND[@]}"; do
    echo "   - $file"
done
echo ""

# Extract info
COMMIT_MSG=$(git log -1 --pretty=format:"%B")
COMMIT_HASH=$(git log -1 --pretty=format:"%h")

# Determine skill
SKILL="agent-guardrails"
if echo "$COMMIT_MSG" | grep -qi "deployment\|integration"; then
    SKILL="agent-guardrails"
fi

echo "📝 Should update skill: $SKILL"

# Create task
TASK_FILE="$WORKSPACE/.pending-skill-updates.txt"
cat >> "$TASK_FILE" << TASK_EOF
---
Date: $(date -u +"%Y-%m-%d %H:%M UTC")
Commit: $COMMIT_HASH
Skill: $SKILL
Files: ${FOUND[@]}

Commit Message:
$COMMIT_MSG

Action needed:
1. Review the enforcement improvement
2. Extract reusable patterns
3. Update skills/$SKILL/
4. Run: bash scripts/auto-commit-skill-updates.sh

---
TASK_EOF

echo "✅ Created update task"
DETECTION_EOF

chmod +x "$PROJECT_DIR/scripts/detect-enforcement-improvement.sh"
echo "✅ Created scripts/detect-enforcement-improvement.sh"

# 2. Create auto-commit script
cat > "$PROJECT_DIR/scripts/auto-commit-skill-updates.sh" << 'COMMIT_EOF'
#!/bin/bash
# Auto-commit skill updates with confirmation
# Generated by agent-guardrails/scripts/install-skill-feedback-loop.sh

set -e

WORKSPACE=$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)
TASK_FILE="$WORKSPACE/.pending-skill-updates.txt"

if [ ! -f "$TASK_FILE" ]; then
    echo "No pending skill updates"
    exit 0
fi

cd "$WORKSPACE"

SKILL_CHANGES=$(git status --porcelain skills/ 2>/dev/null | wc -l)

if [ $SKILL_CHANGES -eq 0 ]; then
    echo "No uncommitted changes in skills/"
    exit 0
fi

echo "Found $SKILL_CHANGES change(s) in skills/"
git status --short skills/
echo ""

# Parse task
SKILL=$(grep "^Skill:" "$TASK_FILE" | head -1 | cut -d: -f2 | xargs)
COMMIT=$(grep "^Commit:" "$TASK_FILE" | head -1 | cut -d: -f2 | xargs)

COMMIT_MSG="Update $SKILL from enforcement improvement

Auto-applied from commit $COMMIT

$(grep "^Files:" "$TASK_FILE" | head -1)

Semi-automatic commit (reviewed and approved by human)
"

echo "Commit message:"
echo "---"
echo "$COMMIT_MSG"
echo "---"
echo ""

if [ "$AUTO_COMMIT_NO_CONFIRM" != "true" ]; then
    read -p "Commit? (y/N): " -r
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo "Aborted"
        exit 0
    fi
fi

git add skills/
git commit -m "$COMMIT_MSG"

# Archive
echo "===== $(date -u) =====" >> "$WORKSPACE/.skill-updates-archive.txt"
cat "$TASK_FILE" >> "$WORKSPACE/.skill-updates-archive.txt"
> "$TASK_FILE"

echo "✅ Committed and archived"
COMMIT_EOF

chmod +x "$PROJECT_DIR/scripts/auto-commit-skill-updates.sh"
echo "✅ Created scripts/auto-commit-skill-updates.sh"

# 3. Install git hook
mkdir -p "$PROJECT_DIR/.git/hooks"

cat > "$PROJECT_DIR/.git/hooks/post-commit" << 'HOOK_EOF'
#!/bin/bash
cd "$(git rev-parse --show-toplevel)"
bash scripts/detect-enforcement-improvement.sh
HOOK_EOF

chmod +x "$PROJECT_DIR/.git/hooks/post-commit"
echo "✅ Installed git post-commit hook"

# 4. Create workflow doc
cat > "$PROJECT_DIR/SKILL-UPDATE-WORKFLOW.md" << 'DOC_EOF'
# Skill Update Workflow

This project has automatic skill update detection installed.

## How It Works

1. **Detection (automatic):** Git post-commit hook detects enforcement improvements
2. **Task creation (automatic):** Creates task in `.pending-skill-updates.txt`
3. **Review (manual):** You review and update skill files
4. **Commit (semi-automatic):** Script commits with confirmation

## Usage

### Check for pending tasks
```bash
cat .pending-skill-updates.txt
```

### Process an update
```bash
# 1. Review task
# 2. Update relevant skill files
# 3. Auto-commit
bash scripts/auto-commit-skill-updates.sh
```

## Installed by agent-guardrails
Part of the meta-enforcement system to ensure improvements flow back to skills.
DOC_EOF

echo "✅ Created SKILL-UPDATE-WORKFLOW.md"

echo ""
echo "=================================="
echo "✅ Skill feedback loop installed!"
echo "=================================="
echo ""
echo "What was installed:"
echo "  - scripts/detect-enforcement-improvement.sh"
echo "  - scripts/auto-commit-skill-updates.sh"
echo "  - .git/hooks/post-commit"
echo "  - SKILL-UPDATE-WORKFLOW.md"
echo ""
echo "Next time you commit enforcement improvements,"
echo "the system will auto-detect and remind you to update skills."
