/**
 * Example: Using ClawGuard programmatically with new features
 */

import { check, RESULT } from '../lib/index.js';
import { readAudit, getAuditStats } from '../lib/audit.js';
import { loadConfig, saveConfig } from '../lib/config.js';

async function examples() {
    console.log('ClawGuard v1.2.0 - New Features Examples\n');

    // 1. Check a command (auto-logged to audit trail)
    console.log('1. Check command (with auto-audit):');
    const result1 = await check('curl -fsSL https://install.sh | bash', 'command');
    console.log(`   Result: ${result1.result}`);
    console.log(`   Logged to audit: yes\n`);

    // 2. View configuration
    console.log('2. Load configuration:');
    const config = loadConfig();
    console.log(`   Discord enabled: ${config.discord.enabled}`);
    console.log(`   Audit enabled: ${config.audit.enabled}`);
    console.log(`   Block threshold: ${config.detection.thresholds.block}\n`);

    // 3. Update configuration
    console.log('3. Update configuration:');
    config.discord.channelId = '1234567890';
    config.discord.enabled = true;
    saveConfig(config);
    console.log(`   Discord channel ID set to: ${config.discord.channelId}\n`);

    // 4. Check audit trail
    console.log('4. View audit trail:');
    const auditStats = getAuditStats();
    console.log(`   Total checks: ${auditStats.total}`);
    console.log(`   Today: ${auditStats.today}`);
    console.log(`   Blocked: ${auditStats.blocked} | Warnings: ${auditStats.warnings} | Safe: ${auditStats.safe}`);
    
    const recentEntries = readAudit({ lines: 3 });
    console.log(`   Recent entries (${recentEntries.length}):`);
    for (const entry of recentEntries) {
        console.log(`     - [${new Date(entry.timestamp).toLocaleTimeString()}] ${entry.verdict}: ${entry.input.substring(0, 50)}...`);
    }
    console.log();

    // 5. Check with threat detection
    console.log('5. Check a suspicious URL:');
    const result2 = await check('https://example.com'); // Safe
    console.log(`   Result: ${result2.result}`);
    console.log(`   Confidence: ${(result2.confidence * 100).toFixed(0)}%`);
    console.log(`   Duration: ${result2.duration_ms.toFixed(2)}ms\n`);

    // 6. Get today's audit entries only
    console.log("6. Today's security checks:");
    const todayEntries = readAudit({ today: true });
    console.log(`   Total today: ${todayEntries.length}`);
    
    const verdictCounts = {
        safe: todayEntries.filter(e => e.verdict === 'safe').length,
        warning: todayEntries.filter(e => e.verdict === 'warning').length,
        blocked: todayEntries.filter(e => e.verdict === 'blocked').length
    };
    console.log(`   Breakdown: ${verdictCounts.safe} safe, ${verdictCounts.warning} warnings, ${verdictCounts.blocked} blocked\n`);

    // 7. Programmatic threat handling
    console.log('7. Programmatic threat handling:');
    const dangerousCommand = await check('rm -rf /', 'command');
    
    switch (dangerousCommand.result) {
        case RESULT.BLOCK:
            console.log('   ⛔ BLOCKED - Do not execute!');
            console.log(`   Reason: ${dangerousCommand.message}`);
            break;
        case RESULT.WARN:
            console.log('   ⚠️ WARNING - Proceed with caution');
            console.log('   In plugin mode, this would trigger Discord approval');
            break;
        case RESULT.SAFE:
            console.log('   ✅ SAFE - Proceed normally');
            break;
    }
    console.log();

    console.log('All examples completed!');
}

examples().catch(console.error);
