---
name: JSON
description: Work with JSON data structures, APIs, and serialization effectively.
metadata: {"clawdbot":{"emoji":"📦","os":["linux","darwin","win32"]}}
---

## Schema & Validation

- Always validate against JSON Schema before processing untrusted input—don't assume structure
- Define schemas for API responses—catches contract violations early
- Use `additionalProperties: false` to reject unknown fields in strict contexts

## Naming & Consistency

- Pick one convention and stick to it—`camelCase` for JS ecosystems, `snake_case` for Python/Ruby
- Avoid mixed conventions in same payload—`userId` alongside `user_name` confuses consumers
- Use plural for collections: `"users": []` not `"user": []`

## Null Handling

- Distinguish "field is null" from "field is absent"—they mean different things
- Omit optional fields entirely rather than sending `null`—reduces payload, clearer intent
- Document which fields are nullable in schema—don't surprise consumers

## Dates & Times

- Always use ISO 8601: `"2024-01-15T14:30:00Z"`—no ambiguous formats like `"01/15/24"`
- Include timezone or use UTC with `Z` suffix—local times without zone are useless
- Timestamps as strings, not epoch integers—human-readable, no precision loss

## Numbers & IDs

- Large IDs as strings: `"id": "9007199254740993"`—JavaScript loses precision above 2^53
- Money as string or integer cents—never float: `"price": "19.99"` or `"price_cents": 1999`
- Avoid floats for anything requiring exactness—currency, coordinates with precision

## Structure Best Practices

- Keep nesting shallow—3 levels max; flatten or split into related endpoints
- Consistent envelope for APIs: `{"data": ..., "meta": ..., "errors": ...}`
- Paginate large arrays—never return unbounded lists; include `next`/`prev` links or cursor

## API Response Patterns

- Errors as structured objects: `{"code": "INVALID_EMAIL", "message": "...", "field": "email"}`
- Include request ID in responses for debugging: `"request_id": "abc-123"`
- Return created/updated resource in response—saves client a follow-up GET

## Serialization

- `toJSON()` method silently overrides output—Date becomes string, custom classes may surprise
- Map, Set, BigInt don't serialize—need custom replacer function
- Circular references throw—detect cycles before stringify or use libraries like `flatted`
- Strip sensitive data before serializing—don't rely on client to ignore extra fields

## Parsing Safety

- `__proto__` key can pollute prototypes—sanitize input or use `Object.create(null)`
- Parse in try/catch—malformed JSON from external sources is common
- Reviver function for type reconstruction: dates, BigInt, custom types

## Unicode

- Emoji need surrogate pairs in escapes: 😀 = `\uD83D\uDE00`—single `\u1F600` invalid
- Control chars U+0000–U+001F must be escaped—pasted text may contain invisible ones
- BOM at file start breaks parsing—strip `\uFEFF` from file input
