---
name: autonomous-skill-orchestrator
description: >
  Deterministically coordinates autonomous planning and execution across available skills under
  strict guardrails. Use only when the user explicitly activates this skill by name to run
  autonomously until a stop command is issued. Trigger keywords include: "use autonomous-skill-orchestrator",
  "activate autonomous-skill-orchestrator", "start autonomous orchestration".
metadata:
  version: "2.0.0"
  owner: "user"
  inspired_by: "oh-my-opencode (Sisyphus, Atlas, Prometheus)"
---

# Autonomous Skill Orchestrator v2.0

> Inspired by oh-my-opencode's three-layer architecture, adapted for OpenClaw's ecosystem.

## Core Philosophy

Traditional AI follows: user asks → AI responds. This fails for complex work because:
1. **Context overload**: Large tasks exceed context windows
2. **Cognitive drift**: AI loses track mid-task
3. **Verification gaps**: No systematic completeness check
4. **Human bottleneck**: Requires constant intervention

This skill solves these through **specialization and delegation**.

---

## Architecture

```
┌─────────────────────────────────────────────────────────┐
│  PLANNING LAYER (Interview + Plan Generation)          │
│  • Clarify intent through interview                     │
│  • Generate structured work plan                        │
│  • Review plan for gaps                                 │
└─────────────────────────────────────────────────────────┘
                          ↓
┌─────────────────────────────────────────────────────────┐
│  ORCHESTRATION LAYER (Atlas - The Conductor)           │
│  • Read plan, delegate tasks                            │
│  • Accumulate wisdom across tasks                       │
│  • Verify results independently                         │
│  • NEVER write code directly — only delegate            │
└─────────────────────────────────────────────────────────┘
                          ↓
┌─────────────────────────────────────────────────────────┐
│  EXECUTION LAYER (Sub-agents via sessions_spawn)       │
│  • Focused task execution                               │
│  • Return results + learnings                           │
│  • Isolated context per task                            │
└─────────────────────────────────────────────────────────┘
```

---

## Activation

### Explicit Triggers
- "use autonomous-skill-orchestrator"
- "activate autonomous-skill-orchestrator"
- "start autonomous orchestration"
- "ulw" or "ultrawork" (magic keyword mode)

### Magic Word: `ultrawork` / `ulw`
Include `ultrawork` or `ulw` in any prompt to activate full orchestration mode automatically.
The agent figures out the rest — parallel agents, background tasks, deep exploration, and relentless execution until completion.

---

## Phase 1: Planning (Prometheus Mode)

### Step 1.1: Interview
Before planning, gather clarity through brief interview:

**Ask only what's needed:**
- What's the core objective?
- What are the boundaries (what's NOT in scope)?
- Any constraints or preferences?
- How do we know when it's done?

**Interview Style by Intent:**
| Intent | Focus | Example Questions |
|--------|-------|-------------------|
| **Refactoring** | Safety | "What tests verify current behavior?" |
| **Build New** | Patterns | "Follow existing conventions or deviate?" |
| **Debug/Fix** | Reproduction | "Steps to reproduce? Error messages?" |
| **Research** | Scope | "Depth vs breadth? Time constraints?" |

### Step 1.2: Plan Generation
After interview, generate structured plan:

```markdown
## Work Plan: [Title]

### Objective
[One sentence, frozen intent]

### Tasks
- [ ] Task 1: [Description]
  - Acceptance: [How to verify completion]
  - References: [Files, docs, skills needed]
  - Category: [quick|general|deep|creative]
  
- [ ] Task 2: ...

### Guardrails
- MUST: [Required constraints]
- MUST NOT: [Forbidden actions]

### Verification
[How to verify overall completion]
```

### Step 1.3: Plan Review (Self-Momus)
Before execution, validate:
- [ ] Each task has clear acceptance criteria
- [ ] References are concrete (not vague)
- [ ] No scope creep beyond objective
- [ ] Dependencies between tasks are explicit
- [ ] Guardrails are actionable

If any check fails, refine plan before proceeding.

---

## Phase 2: Orchestration (Atlas Mode)

### Conductor Rules
The orchestrator:
- ✅ CAN read files to understand context
- ✅ CAN run commands to verify results
- ✅ CAN search patterns with grep/glob
- ✅ CAN spawn sub-agents for work

The orchestrator:
- ❌ MUST NOT write/edit code directly
- ❌ MUST NOT trust sub-agent claims blindly
- ❌ MUST NOT skip verification

### Step 2.1: Task Delegation

Use `sessions_spawn` with category-appropriate configuration:

| Category | Use For | Model Hint | Timeout |
|----------|---------|------------|---------|
| `quick` | Trivial tasks, single file changes | fast model | 2-5 min |
| `general` | Standard implementation | default | 5-10 min |
| `deep` | Complex logic, architecture | thinking model | 10-20 min |
| `creative` | UI/UX, content generation | creative model | 5-10 min |
| `research` | Docs, codebase exploration | fast + broad | 5 min |

**Delegation Template:**
```
sessions_spawn(
  label: "task-{n}-{short-desc}",
  task: """
  ## Task
  {exact task from plan}
  
  ## Expected Outcome
  {acceptance criteria}
  
  ## Context
  {accumulated wisdom from previous tasks}
  
  ## Constraints
  - MUST: {guardrails}
  - MUST NOT: {forbidden actions}
  
  ## References
  {relevant files, docs}
  """,
  runTimeoutSeconds: {based on category}
)
```

### Step 2.2: Parallel Execution

Identify independent tasks (no file conflicts, no dependencies) and spawn them simultaneously:

```
# Tasks 2, 3, 4 have no dependencies
sessions_spawn(label="task-2", task="...")
sessions_spawn(label="task-3", task="...")
sessions_spawn(label="task-4", task="...")
# All run in parallel
```

### Step 2.3: Wisdom Accumulation

After each task completion, extract and record:

```markdown
## Wisdom Log

### Conventions Discovered
- [Pattern found in codebase]

### Successful Approaches
- [What worked]

### Gotchas
- [Pitfalls to avoid]

### Commands Used
- [Useful commands for similar tasks]
```

Store in: `memory/orchestrator-wisdom.md` (append-only during session)

Pass accumulated wisdom to ALL subsequent sub-agents.

### Step 2.4: Independent Verification

**NEVER trust sub-agent claims.** After each task:
1. Read actual changed files
2. Run tests/linting if applicable
3. Verify acceptance criteria independently
4. Cross-reference with plan requirements

If verification fails:
- Log the failure in wisdom
- Re-delegate with failure context
- Max 2 retries per task, then escalate to user

---

## Phase 3: Completion

### Step 3.1: Final Verification
- All tasks marked complete
- All acceptance criteria verified
- No unresolved issues in wisdom log

### Step 3.2: Summary Report
```markdown
## Orchestration Complete

### Completed Tasks
- [x] Task 1: {summary}
- [x] Task 2: {summary}

### Learnings
{key wisdom accumulated}

### Files Changed
{list of modified files}

### Next Steps (if any)
{recommendations}
```

---

## Safety Guardrails

### Halt Conditions (Immediate Stop)
- User issues explicit stop command
- Irreversible destructive action detected
- Scope expansion beyond frozen intent
- 3+ consecutive task failures
- Sub-agent attempts to spawn further sub-agents (no recursion)

### Risk Classification
| Class | Description | Action |
|-------|-------------|--------|
| A | Irreversible, destructive, or unbounded | HALT immediately |
| B | Bounded, resolvable with clarification | Pause, ask user |
| C | Cosmetic, non-operative | Proceed with note |

### Forbidden Actions
- Creating new autonomous orchestrators
- Modifying this skill file
- Accessing credentials without explicit need
- External API calls not in original scope
- Recursive spawning (sub-agents spawning sub-agents)

---

## Stop Commands
User can stop at any time with:
- "stop"
- "halt"
- "cancel orchestration"
- "abort"

On stop: immediately terminate all spawned sessions, output summary of completed work, await new instructions.

---

## Memory Integration

### During Orchestration
- Append to `memory/orchestrator-wisdom.md` for learnings
- Reference existing memory files for context

### After Orchestration
- Update daily memory with orchestration summary
- Persist significant learnings to MEMORY.md if valuable

---

## Example Usage

**Simple (magic word):**
```
ulw refactor the authentication module to use JWT
```

**Explicit activation:**
```
activate autonomous-skill-orchestrator

Build a REST API with user registration, login, and profile endpoints
```

**With constraints:**
```
use autonomous-skill-orchestrator
- Build payment integration with Stripe
- MUST: Use existing database patterns
- MUST NOT: Store card numbers locally
- Deadline: Complete core flow only
```
