/**
 * Session management for Janee
 * Tracks active capability sessions with TTL
 */

import { generateToken } from './crypto';
import fs from 'fs';
import path from 'path';
import os from 'os';

export interface Session {
  id: string;
  capability: string;
  service: string;
  agentId?: string;
  reason?: string;
  createdAt: Date;
  expiresAt: Date;
  revoked: boolean;
}

export class SessionManager {
  private sessions: Map<string, Session> = new Map();
  private persistFile: string;

  constructor(persistFile?: string) {
    this.persistFile = persistFile || path.join(os.homedir(), '.janee', 'sessions.json');
    this.load();
  }

  /**
   * Load sessions from file
   */
  private load(): void {
    if (!fs.existsSync(this.persistFile)) {
      return;
    }

    try {
      const data = fs.readFileSync(this.persistFile, 'utf8');
      const sessions = JSON.parse(data) as Array<Omit<Session, 'createdAt' | 'expiresAt'> & {
        createdAt: string;
        expiresAt: string;
      }>;

      for (const s of sessions) {
        this.sessions.set(s.id, {
          ...s,
          createdAt: new Date(s.createdAt),
          expiresAt: new Date(s.expiresAt)
        });
      }
    } catch (error) {
      console.error('Warning: Failed to load sessions:', error);
    }
  }

  /**
   * Save sessions to file
   */
  private save(): void {
    const sessions = Array.from(this.sessions.values()).map(s => ({
      ...s,
      createdAt: s.createdAt.toISOString(),
      expiresAt: s.expiresAt.toISOString()
    }));

    const dir = path.dirname(this.persistFile);
    if (!fs.existsSync(dir)) {
      fs.mkdirSync(dir, { recursive: true, mode: 0o700 });
    }

    fs.writeFileSync(this.persistFile, JSON.stringify(sessions, null, 2), { mode: 0o600 });
  }

  /**
   * Create a new session for a capability
   */
  createSession(
    capability: string,
    service: string,
    ttlSeconds: number,
    options: { agentId?: string; reason?: string } = {}
  ): Session {
    const id = generateToken('jnee_sess', 32);
    const now = new Date();
    const expiresAt = new Date(now.getTime() + ttlSeconds * 1000);

    const session: Session = {
      id,
      capability,
      service,
      agentId: options.agentId,
      reason: options.reason,
      createdAt: now,
      expiresAt,
      revoked: false
    };

    this.sessions.set(id, session);
    this.save();

    // Schedule cleanup
    setTimeout(() => {
      this.sessions.delete(id);
      this.save();
    }, ttlSeconds * 1000);

    return session;
  }

  /**
   * Get a session by ID
   */
  getSession(sessionId: string): Session | undefined {
    const session = this.sessions.get(sessionId);
    
    if (!session) {
      return undefined;
    }

    // Check expiry
    if (session.revoked || new Date() > session.expiresAt) {
      this.sessions.delete(sessionId);
      return undefined;
    }

    return session;
  }

  /**
   * Revoke a session immediately
   */
  revokeSession(sessionId: string): boolean {
    const session = this.sessions.get(sessionId);
    
    if (!session) {
      return false;
    }

    session.revoked = true;
    this.sessions.delete(sessionId);
    this.save();
    return true;
  }

  /**
   * List all active sessions
   */
  listSessions(): Session[] {
    const now = new Date();
    const active: Session[] = [];

    for (const [id, session] of this.sessions.entries()) {
      if (!session.revoked && now <= session.expiresAt) {
        active.push(session);
      } else {
        // Clean up expired
        this.sessions.delete(id);
      }
    }

    return active;
  }

  /**
   * Clean up expired sessions
   */
  cleanup(): void {
    const now = new Date();
    
    for (const [id, session] of this.sessions.entries()) {
      if (session.revoked || now > session.expiresAt) {
        this.sessions.delete(id);
      }
    }
  }
}
