/**
 * Request policy rules engine
 * Enforces allow/deny patterns on API requests
 */

export interface Rules {
  allow?: string[];  // e.g., ["GET *", "POST /v1/refunds/*"]
  deny?: string[];   // e.g., ["POST /v1/charges/*"]
}

export interface RuleCheckResult {
  allowed: boolean;
  reason?: string;
  matchedRule?: string;
}

/**
 * Check if a request matches policy rules
 * 
 * Logic:
 * 1. No rules defined → allow all (backward compatible)
 * 2. Check deny patterns first (explicit deny wins)
 * 3. Check allow patterns
 * 4. If rules defined but no match → deny by default
 */
export function checkRules(
  rules: Rules | undefined,
  method: string,
  path: string
): RuleCheckResult {
  // No rules = allow all (backward compatible)
  if (!rules || (!rules.allow && !rules.deny)) {
    return { allowed: true };
  }
  
  // Check deny first (explicit deny wins)
  if (rules.deny) {
    for (const pattern of rules.deny) {
      if (matchPattern(pattern, method, path)) {
        return { 
          allowed: false, 
          reason: `Denied by rule: ${pattern}`,
          matchedRule: pattern
        };
      }
    }
  }
  
  // Check allow
  if (rules.allow) {
    for (const pattern of rules.allow) {
      if (matchPattern(pattern, method, path)) {
        return { 
          allowed: true,
          matchedRule: pattern
        };
      }
    }
    // Rules defined but no allow match = deny
    return { 
      allowed: false, 
      reason: 'No matching allow rule' 
    };
  }
  
  // Only deny rules, none matched = allow
  return { allowed: true };
}

/**
 * Match a request against a pattern
 * 
 * Pattern format: "METHOD PATH"
 * - METHOD: HTTP method or "*" for any
 * - PATH: Path with optional glob wildcards (*)
 * 
 * Examples:
 * - "GET *" → any GET request
 * - "POST /v1/charges/*" → POST to /v1/charges/ and subpaths
 * - "* /v1/customers" → any method to /v1/customers
 * - "DELETE /v1/customers/*" → DELETE to any customer
 */
export function matchPattern(pattern: string, method: string, path: string): boolean {
  const parts = pattern.trim().split(/\s+/);
  
  if (parts.length !== 2) {
    console.warn(`Invalid rule pattern: "${pattern}" (expected "METHOD PATH")`);
    return false;
  }
  
  const [patternMethod, patternPath] = parts;
  
  // Check method
  if (patternMethod !== '*' && patternMethod.toUpperCase() !== method.toUpperCase()) {
    return false;
  }
  
  // Check path with glob matching
  return matchGlob(patternPath, path);
}

/**
 * Match a path against a glob pattern
 * 
 * Supports:
 * - * → matches any characters
 * - /v1/customers/* → matches /v1/customers/123, /v1/customers/abc, etc.
 * - /v1/balance → exact match
 */
function matchGlob(pattern: string, path: string): boolean {
  // Exact match
  if (pattern === path) {
    return true;
  }
  
  // No wildcards = exact match only
  if (!pattern.includes('*')) {
    return false;
  }
  
  // Convert glob pattern to regex
  // Escape special regex chars except *
  const regexPattern = pattern
    .replace(/[.+?^${}()|[\]\\]/g, '\\$&')  // Escape special chars
    .replace(/\*/g, '.*');                    // * becomes .*
  
  const regex = new RegExp(`^${regexPattern}$`);
  return regex.test(path);
}

/**
 * Validate rules configuration
 * Returns array of validation errors (empty = valid)
 */
export function validateRules(rules: Rules | undefined): string[] {
  if (!rules) return [];
  
  const errors: string[] = [];
  
  // Validate allow patterns
  if (rules.allow) {
    if (!Array.isArray(rules.allow)) {
      errors.push('rules.allow must be an array');
    } else {
      rules.allow.forEach((pattern, i) => {
        if (typeof pattern !== 'string') {
          errors.push(`rules.allow[${i}] must be a string`);
        } else if (!pattern.trim().includes(' ')) {
          errors.push(`rules.allow[${i}] invalid format: "${pattern}" (expected "METHOD PATH")`);
        }
      });
    }
  }
  
  // Validate deny patterns
  if (rules.deny) {
    if (!Array.isArray(rules.deny)) {
      errors.push('rules.deny must be an array');
    } else {
      rules.deny.forEach((pattern, i) => {
        if (typeof pattern !== 'string') {
          errors.push(`rules.deny[${i}] must be a string`);
        } else if (!pattern.trim().includes(' ')) {
          errors.push(`rules.deny[${i}] invalid format: "${pattern}" (expected "METHOD PATH")`);
        }
      });
    }
  }
  
  return errors;
}
