/**
 * Tests for rules engine
 */

import { describe, it, expect } from 'vitest';
import { checkRules, matchPattern, validateRules, Rules } from './rules';

describe('Rules Engine', () => {
  describe('checkRules', () => {
    it('should allow all when no rules defined', () => {
      const result = checkRules(undefined, 'GET', '/v1/customers');
      expect(result.allowed).toBe(true);
    });

    it('should allow all when empty rules', () => {
      const result = checkRules({}, 'POST', '/v1/charges');
      expect(result.allowed).toBe(true);
    });

    it('should allow matching allow rule', () => {
      const rules: Rules = {
        allow: ['GET *']
      };
      const result = checkRules(rules, 'GET', '/v1/customers');
      expect(result.allowed).toBe(true);
      expect(result.matchedRule).toBe('GET *');
    });

    it('should deny non-matching allow rule', () => {
      const rules: Rules = {
        allow: ['GET *']
      };
      const result = checkRules(rules, 'POST', '/v1/charges');
      expect(result.allowed).toBe(false);
      expect(result.reason).toContain('No matching allow rule');
    });

    it('should deny matching deny rule', () => {
      const rules: Rules = {
        deny: ['POST *']
      };
      const result = checkRules(rules, 'POST', '/v1/charges');
      expect(result.allowed).toBe(false);
      expect(result.reason).toContain('Denied by rule');
      expect(result.matchedRule).toBe('POST *');
    });

    it('should allow non-matching deny rule', () => {
      const rules: Rules = {
        deny: ['POST *']
      };
      const result = checkRules(rules, 'GET', '/v1/customers');
      expect(result.allowed).toBe(true);
    });

    it('should check deny before allow (deny wins)', () => {
      const rules: Rules = {
        allow: ['POST *'],
        deny: ['POST /v1/charges/*']
      };
      const result = checkRules(rules, 'POST', '/v1/charges/ch_123');
      expect(result.allowed).toBe(false);
      expect(result.matchedRule).toBe('POST /v1/charges/*');
    });

    it('should allow when deny does not match but allow does', () => {
      const rules: Rules = {
        allow: ['POST *'],
        deny: ['POST /v1/charges/*']
      };
      const result = checkRules(rules, 'POST', '/v1/refunds');
      expect(result.allowed).toBe(true);
    });
  });

  describe('matchPattern', () => {
    it('should match exact method and path', () => {
      expect(matchPattern('GET /v1/balance', 'GET', '/v1/balance')).toBe(true);
    });

    it('should match wildcard method', () => {
      expect(matchPattern('* /v1/balance', 'GET', '/v1/balance')).toBe(true);
      expect(matchPattern('* /v1/balance', 'POST', '/v1/balance')).toBe(true);
    });

    it('should match wildcard path', () => {
      expect(matchPattern('GET *', 'GET', '/v1/customers')).toBe(true);
      expect(matchPattern('GET *', 'GET', '/v1/balance')).toBe(true);
      expect(matchPattern('GET *', 'GET', '/')).toBe(true);
    });

    it('should match path with trailing wildcard', () => {
      expect(matchPattern('POST /v1/charges/*', 'POST', '/v1/charges/ch_123')).toBe(true);
      expect(matchPattern('POST /v1/charges/*', 'POST', '/v1/charges/')).toBe(true);
    });

    it('should not match different method', () => {
      expect(matchPattern('GET /v1/balance', 'POST', '/v1/balance')).toBe(false);
    });

    it('should not match different path', () => {
      expect(matchPattern('GET /v1/balance', 'GET', '/v1/customers')).toBe(false);
    });

    it('should not match when wildcard does not cover', () => {
      expect(matchPattern('POST /v1/charges/*', 'POST', '/v1/refunds')).toBe(false);
    });

    it('should be case-insensitive for methods', () => {
      expect(matchPattern('get /v1/balance', 'GET', '/v1/balance')).toBe(true);
      expect(matchPattern('GET /v1/balance', 'get', '/v1/balance')).toBe(true);
    });

    it('should handle multiple wildcards', () => {
      expect(matchPattern('GET /v1/*/metadata', 'GET', '/v1/customers/metadata')).toBe(true);
      expect(matchPattern('GET /v1/*/metadata', 'GET', '/v1/charges/metadata')).toBe(true);
    });

    it('should return false for invalid pattern format', () => {
      expect(matchPattern('GET', 'GET', '/v1/balance')).toBe(false);
      expect(matchPattern('/v1/balance', 'GET', '/v1/balance')).toBe(false);
    });
  });

  describe('validateRules', () => {
    it('should return no errors for undefined rules', () => {
      const errors = validateRules(undefined);
      expect(errors).toEqual([]);
    });

    it('should return no errors for empty rules', () => {
      const errors = validateRules({});
      expect(errors).toEqual([]);
    });

    it('should return no errors for valid rules', () => {
      const rules: Rules = {
        allow: ['GET *', 'POST /v1/refunds/*'],
        deny: ['DELETE *']
      };
      const errors = validateRules(rules);
      expect(errors).toEqual([]);
    });

    it('should error if allow is not an array', () => {
      const rules = { allow: 'GET *' } as any;
      const errors = validateRules(rules);
      expect(errors).toContain('rules.allow must be an array');
    });

    it('should error if deny is not an array', () => {
      const rules = { deny: 'POST *' } as any;
      const errors = validateRules(rules);
      expect(errors).toContain('rules.deny must be an array');
    });

    it('should error for invalid pattern format in allow', () => {
      const rules: Rules = {
        allow: ['GET', 'POST /v1/charges/*']
      };
      const errors = validateRules(rules);
      expect(errors.length).toBeGreaterThan(0);
      expect(errors[0]).toContain('invalid format');
    });

    it('should error for invalid pattern format in deny', () => {
      const rules: Rules = {
        deny: ['/v1/charges/*']
      };
      const errors = validateRules(rules);
      expect(errors.length).toBeGreaterThan(0);
      expect(errors[0]).toContain('invalid format');
    });

    it('should error for non-string patterns', () => {
      const rules = {
        allow: [123, 'GET *']
      } as any;
      const errors = validateRules(rules);
      expect(errors.length).toBeGreaterThan(0);
      expect(errors[0]).toContain('must be a string');
    });
  });
});
