/**
 * MCP Server for Janee
 * Exposes capabilities to AI agents via Model Context Protocol
 */

import { Server } from '@modelcontextprotocol/sdk/server/index.js';
import { StdioServerTransport } from '@modelcontextprotocol/sdk/server/stdio.js';
import {
  CallToolRequestSchema,
  ListToolsRequestSchema,
  Tool
} from '@modelcontextprotocol/sdk/types.js';
import { SessionManager } from './sessions.js';
import { checkRules, Rules } from './rules.js';
import { AuditLogger } from './audit.js';
import http from 'http';
import https from 'https';
import { URL } from 'url';

export interface Capability {
  name: string;
  service: string;
  ttl: string;  // e.g., "1h", "30m"
  autoApprove?: boolean;
  requiresReason?: boolean;
  rules?: Rules;  // Optional allow/deny patterns
}

export interface ServiceConfig {
  baseUrl: string;
  auth: {
    type: 'bearer' | 'hmac' | 'headers';
    key?: string;
    apiKey?: string;
    apiSecret?: string;
    headers?: Record<string, string>;
  };
}

export interface APIRequest {
  service: string;
  path: string;
  method: string;
  headers: Record<string, string>;
  body?: string;
}

export interface APIResponse {
  statusCode: number;
  headers: Record<string, string | string[]>;
  body: string;
}

export interface ReloadResult {
  capabilities: Capability[];
  services: Map<string, ServiceConfig>;
}

export interface MCPServerOptions {
  capabilities: Capability[];
  services: Map<string, ServiceConfig>;
  sessionManager: SessionManager;
  auditLogger: AuditLogger;
  onExecute: (session: any, request: APIRequest) => Promise<APIResponse>;
  onReloadConfig?: () => ReloadResult;
}

/**
 * Parse TTL string to seconds
 */
function parseTTL(ttl: string): number {
  const match = ttl.match(/^(\d+)([smhd])$/);
  if (!match) throw new Error(`Invalid TTL format: ${ttl}`);
  
  const value = parseInt(match[1]);
  const unit = match[2];
  
  const multipliers: Record<string, number> = {
    s: 1,
    m: 60,
    h: 3600,
    d: 86400
  };
  
  return value * multipliers[unit];
}

/**
 * Create and start MCP server
 */
export function createMCPServer(options: MCPServerOptions): Server {
  const { sessionManager, auditLogger, onExecute, onReloadConfig } = options;
  
  // Store as mutable to support hot-reloading
  let capabilities = options.capabilities;
  let services = options.services;

  const server = new Server(
    {
      name: 'janee',
      version: '0.1.0'
    },
    {
      capabilities: {
        tools: {}
      }
    }
  );

  // Tool: list_services
  const listServicesTool: Tool = {
    name: 'list_services',
    description: 'List available API capabilities managed by Janee',
    inputSchema: {
      type: 'object',
      properties: {},
      required: []
    }
  };

  // Tool: execute
  const executeTool: Tool = {
    name: 'execute',
    description: 'Execute an API request through Janee proxy',
    inputSchema: {
      type: 'object',
      properties: {
        capability: {
          type: 'string',
          description: 'Capability name to use (from list_services)'
        },
        method: {
          type: 'string',
          enum: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH'],
          description: 'HTTP method'
        },
        path: {
          type: 'string',
          description: 'API path (e.g., /v1/customers)'
        },
        body: {
          type: 'string',
          description: 'Request body (JSON string, optional)'
        },
        headers: {
          type: 'object',
          description: 'Additional headers (optional)',
          additionalProperties: { type: 'string' }
        },
        reason: {
          type: 'string',
          description: 'Reason for this request (required for some capabilities)'
        }
      },
      required: ['capability', 'method', 'path']
    }
  };

  // Tool: reload_config
  const reloadConfigTool: Tool = {
    name: 'reload_config',
    description: 'Reload Janee configuration from disk without restarting the server. Use after adding new services or capabilities.',
    inputSchema: {
      type: 'object',
      properties: {},
      required: []
    }
  };

  // Register tools
  server.setRequestHandler(ListToolsRequestSchema, async () => {
    const tools = [listServicesTool, executeTool];
    // Only expose reload_config if a reload handler is provided
    if (onReloadConfig) {
      tools.push(reloadConfigTool);
    }
    return { tools };
  });

  // Handle tool calls
  server.setRequestHandler(CallToolRequestSchema, async (request) => {
    const { name, arguments: args } = request.params;

    try {
      switch (name) {
        case 'list_services':
          return {
            content: [{
              type: 'text',
              text: JSON.stringify(
                capabilities.map(cap => ({
                  name: cap.name,
                  service: cap.service,
                  ttl: cap.ttl,
                  autoApprove: cap.autoApprove,
                  requiresReason: cap.requiresReason,
                  rules: cap.rules
                })),
                null,
                2
              )
            }]
          };

        case 'reload_config': {
          if (!onReloadConfig) {
            throw new Error('Config reload not supported');
          }

          try {
            const result = onReloadConfig();
            const prevCapCount = capabilities.length;
            const prevServiceCount = services.size;
            
            capabilities = result.capabilities;
            services = result.services;

            return {
              content: [{
                type: 'text',
                text: JSON.stringify({
                  success: true,
                  message: 'Configuration reloaded successfully',
                  services: services.size,
                  capabilities: capabilities.length,
                  changes: {
                    services: services.size - prevServiceCount,
                    capabilities: capabilities.length - prevCapCount
                  }
                }, null, 2)
              }]
            };
          } catch (error) {
            throw new Error(`Failed to reload config: ${error instanceof Error ? error.message : 'Unknown error'}`);
          }
        }

        case 'execute': {
          const { capability, method, path, body, headers, reason } = args as any;

          // Find capability
          const cap = capabilities.find(c => c.name === capability);
          if (!cap) {
            throw new Error(`Unknown capability: ${capability}`);
          }

          // Check if reason required
          if (cap.requiresReason && !reason) {
            throw new Error(`Capability "${capability}" requires a reason`);
          }

          // Check rules (path-based policies)
          const ruleCheck = checkRules(cap.rules, method, path);
          if (!ruleCheck.allowed) {
            // Log denied request
            auditLogger.logDenied(
              cap.service,
              method,
              path,
              ruleCheck.reason || 'Request denied by policy',
              reason
            );
            throw new Error(ruleCheck.reason || 'Request denied by policy');
          }

          // Get or create session
          const ttlSeconds = parseTTL(cap.ttl);
          const session = sessionManager.createSession(
            cap.name,
            cap.service,
            ttlSeconds,
            { reason }
          );

          // Build API request
          const apiReq: APIRequest = {
            service: cap.service,
            path,
            method,
            headers: headers || {},
            body
          };

          // Execute
          const response = await onExecute(session, apiReq);

          return {
            content: [{
              type: 'text',
              text: JSON.stringify({
                status: response.statusCode,
                body: response.body
              }, null, 2)
            }]
          };
        }

        default:
          throw new Error(`Unknown tool: ${name}`);
      }
    } catch (error) {
      return {
        content: [{
          type: 'text',
          text: JSON.stringify({
            error: error instanceof Error ? error.message : 'Unknown error'
          }, null, 2)
        }],
        isError: true
      };
    }
  });

  return server;
}

/**
 * Make HTTP/HTTPS request to real API
 */
export function makeAPIRequest(
  targetUrl: URL,
  request: APIRequest
): Promise<APIResponse> {
  return new Promise((resolve, reject) => {
    const client = targetUrl.protocol === 'https:' ? https : http;

    const options = {
      method: request.method,
      headers: request.headers
    };

    const req = client.request(targetUrl, options, (res) => {
      let body = '';

      res.on('data', (chunk) => {
        body += chunk;
      });

      res.on('end', () => {
        resolve({
          statusCode: res.statusCode || 500,
          headers: res.headers as Record<string, string | string[]>,
          body
        });
      });
    });

    req.on('error', (error) => {
      reject(error);
    });

    if (request.body) {
      req.write(request.body);
    }

    req.end();
  });
}

/**
 * Start MCP server with stdio transport
 */
export async function startMCPServer(server: Server): Promise<void> {
  const transport = new StdioServerTransport();
  await server.connect(transport);
  
  console.error('Janee MCP server started');
}
