/**
 * Encryption utilities for Janee
 * Uses Node.js crypto with AES-256-GCM
 */

import crypto from 'crypto';

const ALGORITHM = 'aes-256-gcm';
const IV_LENGTH = 12; // GCM standard
const AUTH_TAG_LENGTH = 16;
const KEY_LENGTH = 32; // 256 bits

/**
 * Generate a new master encryption key
 */
export function generateMasterKey(): string {
  return crypto.randomBytes(KEY_LENGTH).toString('base64');
}

/**
 * Encrypt a secret using AES-256-GCM
 * Returns base64-encoded: iv + authTag + ciphertext
 */
export function encryptSecret(plaintext: string, masterKey: string): string {
  // Decode master key
  const keyBuffer = Buffer.from(masterKey, 'base64');
  
  if (keyBuffer.length !== KEY_LENGTH) {
    throw new Error('Invalid master key length');
  }

  // Generate random IV
  const iv = crypto.randomBytes(IV_LENGTH);

  // Create cipher
  const cipher = crypto.createCipheriv(ALGORITHM, keyBuffer, iv);

  // Encrypt
  const encrypted = Buffer.concat([
    cipher.update(plaintext, 'utf8'),
    cipher.final()
  ]);

  // Get auth tag
  const authTag = cipher.getAuthTag();

  // Combine: iv + authTag + encrypted
  const combined = Buffer.concat([iv, authTag, encrypted]);

  return combined.toString('base64');
}

/**
 * Decrypt a secret using AES-256-GCM
 */
export function decryptSecret(ciphertext: string, masterKey: string): string {
  // Decode master key
  const keyBuffer = Buffer.from(masterKey, 'base64');
  
  if (keyBuffer.length !== KEY_LENGTH) {
    throw new Error('Invalid master key length');
  }

  // Decode ciphertext
  const combined = Buffer.from(ciphertext, 'base64');

  // Extract components
  const iv = combined.subarray(0, IV_LENGTH);
  const authTag = combined.subarray(IV_LENGTH, IV_LENGTH + AUTH_TAG_LENGTH);
  const encrypted = combined.subarray(IV_LENGTH + AUTH_TAG_LENGTH);

  // Create decipher
  const decipher = crypto.createDecipheriv(ALGORITHM, keyBuffer, iv);
  decipher.setAuthTag(authTag);

  // Decrypt
  const decrypted = Buffer.concat([
    decipher.update(encrypted),
    decipher.final()
  ]);

  return decrypted.toString('utf8');
}

/**
 * Hash a string using SHA-256
 */
export function hashString(input: string): string {
  return crypto.createHash('sha256').update(input).digest('hex');
}

/**
 * Generate a cryptographically secure random token
 */
export function generateToken(prefix: string = '', length: number = 32): string {
  const token = crypto.randomBytes(length).toString('hex');
  return prefix ? `${prefix}_${token}` : token;
}
