/**
 * Audit logging for Janee
 * Logs every API request to local files
 */

import fs from 'fs';
import path from 'path';

export interface APIRequest {
  service: string;
  path: string;
  method: string;
  headers: Record<string, string>;
  body?: string;
}

export interface APIResponse {
  statusCode: number;
  headers: Record<string, string | string[]>;
  body: string;
}

export interface AuditEvent {
  id: string;
  timestamp: string;
  service: string;
  method: string;
  path: string;
  statusCode?: number;
  duration?: number;
  reason?: string;
  agentId?: string;
  denied?: boolean;
  denyReason?: string;
}

export class AuditLogger {
  private logDir: string;
  private currentLogFile: string;

  constructor(logDir: string) {
    this.logDir = logDir;
    this.currentLogFile = this.getLogFilePath();

    // Ensure log directory exists
    if (!fs.existsSync(logDir)) {
      fs.mkdirSync(logDir, { recursive: true, mode: 0o700 });
    }
  }

  /**
   * Get current log file path (one file per day)
   */
  private getLogFilePath(): string {
    const date = new Date().toISOString().split('T')[0]; // YYYY-MM-DD
    return path.join(this.logDir, `${date}.jsonl`);
  }

  /**
   * Log an API request
   */
  log(req: APIRequest, res?: APIResponse, duration?: number): void {
    const event: AuditEvent = {
      id: this.generateId(),
      timestamp: new Date().toISOString(),
      service: req.service,
      method: req.method,
      path: req.path,
      statusCode: res?.statusCode,
      duration,
      // TODO: Extract reason/agentId from request headers if present
    };

    // Append to log file (JSONL format)
    const logLine = JSON.stringify(event) + '\n';
    
    // Check if we need to rotate to a new file
    const currentFile = this.getLogFilePath();
    if (currentFile !== this.currentLogFile) {
      this.currentLogFile = currentFile;
    }

    fs.appendFileSync(this.currentLogFile, logLine, { mode: 0o600 });
  }

  /**
   * Log a denied request (blocked by rules)
   */
  logDenied(service: string, method: string, path: string, denyReason: string, userReason?: string): void {
    const event: AuditEvent = {
      id: this.generateId(),
      timestamp: new Date().toISOString(),
      service,
      method,
      path,
      denied: true,
      denyReason,
      reason: userReason,
      statusCode: 403
    };

    // Append to log file (JSONL format)
    const logLine = JSON.stringify(event) + '\n';
    
    // Check if we need to rotate to a new file
    const currentFile = this.getLogFilePath();
    if (currentFile !== this.currentLogFile) {
      this.currentLogFile = currentFile;
    }

    fs.appendFileSync(this.currentLogFile, logLine, { mode: 0o600 });
  }

  /**
   * Read recent logs
   */
  async readLogs(options: {
    limit?: number;
    service?: string;
    since?: Date;
  } = {}): Promise<AuditEvent[]> {
    const { limit = 100, service, since } = options;

    // Get all log files, sorted by date (newest first)
    const files = fs.readdirSync(this.logDir)
      .filter(f => f.endsWith('.jsonl'))
      .sort()
      .reverse();

    const events: AuditEvent[] = [];

    for (const file of files) {
      const filePath = path.join(this.logDir, file);
      const content = fs.readFileSync(filePath, 'utf8');
      const lines = content.trim().split('\n').filter(Boolean);

      for (const line of lines.reverse()) {
        try {
          const event = JSON.parse(line) as AuditEvent;

          // Apply filters
          if (service && event.service !== service) continue;
          if (since && new Date(event.timestamp) < since) continue;

          events.push(event);

          if (events.length >= limit) {
            return events;
          }
        } catch (error) {
          // Skip invalid lines
          console.error('Invalid log line:', error);
        }
      }
    }

    return events;
  }

  /**
   * Follow logs in real-time (tail -f style)
   */
  async *tail(): AsyncGenerator<AuditEvent> {
    // Start from end of current file
    let position = fs.existsSync(this.currentLogFile) 
      ? fs.statSync(this.currentLogFile).size 
      : 0;

    while (true) {
      // Check if log file has grown
      const currentSize = fs.existsSync(this.currentLogFile)
        ? fs.statSync(this.currentLogFile).size
        : 0;

      if (currentSize > position) {
        // Read new content
        const fd = fs.openSync(this.currentLogFile, 'r');
        const buffer = Buffer.alloc(currentSize - position);
        fs.readSync(fd, buffer, 0, buffer.length, position);
        fs.closeSync(fd);

        // Parse new lines
        const content = buffer.toString('utf8');
        const lines = content.split('\n').filter(Boolean);

        for (const line of lines) {
          try {
            const event = JSON.parse(line) as AuditEvent;
            yield event;
          } catch (error) {
            // Skip invalid lines
          }
        }

        position = currentSize;
      }

      // Check if we need to rotate to new file
      const newLogFile = this.getLogFilePath();
      if (newLogFile !== this.currentLogFile) {
        this.currentLogFile = newLogFile;
        position = 0;
      }

      // Wait before checking again
      await new Promise(resolve => setTimeout(resolve, 500));
    }
  }

  /**
   * Generate unique ID for event
   */
  private generateId(): string {
    return Date.now().toString(36) + Math.random().toString(36).substring(2);
  }
}
