/**
 * YAML configuration for Janee (new format)
 * Supports capabilities + services model
 */

import fs from 'fs';
import path from 'path';
import os from 'os';
import yaml from 'js-yaml';
import { encryptSecret, decryptSecret, generateMasterKey } from '../core/crypto';

export interface AuthConfig {
  type: 'bearer' | 'hmac' | 'headers';
  key?: string;
  apiKey?: string;
  apiSecret?: string;
  headers?: Record<string, string>;
}

export interface ServiceConfig {
  baseUrl: string;
  auth: AuthConfig;
}

export interface CapabilityConfig {
  service: string;
  ttl: string;
  autoApprove?: boolean;
  requiresReason?: boolean;
  rules?: {
    allow?: string[];
    deny?: string[];
  };
}

export interface LLMConfig {
  provider?: 'openai' | 'anthropic';
  apiKey?: string;
  model?: string;
}

export interface ServerConfig {
  port: number;
  host: string;
}

export interface JaneeYAMLConfig {
  version: string;
  masterKey: string;
  server: ServerConfig;
  llm?: LLMConfig;
  services: Record<string, ServiceConfig>;
  capabilities: Record<string, CapabilityConfig>;
}

const CONFIG_DIR = path.join(os.homedir(), '.janee');
const CONFIG_FILE_YAML = path.join(CONFIG_DIR, 'config.yaml');
const CONFIG_FILE_JSON = path.join(CONFIG_DIR, 'config.json');
const AUDIT_DIR = path.join(CONFIG_DIR, 'logs');

export function getAuditDir(): string {
  return AUDIT_DIR;
}

export function getConfigDir(): string {
  return CONFIG_DIR;
}

/**
 * Check if using YAML config
 */
export function hasYAMLConfig(): boolean {
  return fs.existsSync(CONFIG_FILE_YAML);
}

/**
 * Load YAML configuration
 */
export function loadYAMLConfig(): JaneeYAMLConfig {
  if (!fs.existsSync(CONFIG_FILE_YAML)) {
    throw new Error('No YAML config found. Run migration or init.');
  }

  const content = fs.readFileSync(CONFIG_FILE_YAML, 'utf8');
  const config = yaml.load(content) as JaneeYAMLConfig;

  // Ensure services and capabilities are objects (YAML parses empty sections as null)
  config.services = config.services || {};
  config.capabilities = config.capabilities || {};

  // Decrypt service auth keys (or use as-is if not encrypted)
  for (const [name, service] of Object.entries(config.services)) {
    const svc = service as ServiceConfig;
    if (svc.auth.type === 'bearer' && svc.auth.key) {
      try {
        svc.auth.key = decryptSecret(svc.auth.key, config.masterKey);
      } catch {
        // Key is plaintext, use as-is
      }
    } else if (svc.auth.type === 'hmac') {
      if (svc.auth.apiKey) {
        try {
          svc.auth.apiKey = decryptSecret(svc.auth.apiKey, config.masterKey);
        } catch {
          // Key is plaintext, use as-is
        }
      }
      if (svc.auth.apiSecret) {
        try {
          svc.auth.apiSecret = decryptSecret(svc.auth.apiSecret, config.masterKey);
        } catch {
          // Key is plaintext, use as-is
        }
      }
    }
  }

  return config;
}

/**
 * Save YAML configuration
 */
export function saveYAMLConfig(config: JaneeYAMLConfig): void {
  // Encrypt service auth keys before saving
  const configCopy = JSON.parse(JSON.stringify(config));

  for (const [name, service] of Object.entries(configCopy.services)) {
    const svc = service as ServiceConfig;
    if (svc.auth.type === 'bearer' && svc.auth.key) {
      svc.auth.key = encryptSecret(svc.auth.key, config.masterKey);
    } else if (svc.auth.type === 'hmac') {
      if (svc.auth.apiKey) {
        svc.auth.apiKey = encryptSecret(svc.auth.apiKey, config.masterKey);
      }
      if (svc.auth.apiSecret) {
        svc.auth.apiSecret = encryptSecret(svc.auth.apiSecret, config.masterKey);
      }
    }
  }

  const yamlContent = yaml.dump(configCopy, {
    indent: 2,
    lineWidth: 120
  });

  fs.writeFileSync(CONFIG_FILE_YAML, yamlContent, { mode: 0o600 });
}

/**
 * Initialize new YAML config
 */
export function initYAMLConfig(): JaneeYAMLConfig {
  if (!fs.existsSync(CONFIG_DIR)) {
    fs.mkdirSync(CONFIG_DIR, { mode: 0o700, recursive: true });
  }

  if (fs.existsSync(CONFIG_FILE_YAML)) {
    throw new Error('Config already exists');
  }

  const config: JaneeYAMLConfig = {
    version: '0.2.0',
    masterKey: generateMasterKey(),
    server: {
      port: 9119,
      host: 'localhost'
    },
    services: {},
    capabilities: {}
  };

  saveYAMLConfig(config);
  return config;
}

/**
 * Add a service to config
 */
export function addServiceYAML(
  name: string,
  baseUrl: string,
  auth: AuthConfig
): void {
  const config = loadYAMLConfig();

  if (config.services[name]) {
    throw new Error(`Service "${name}" already exists`);
  }

  config.services[name] = {
    baseUrl,
    auth
  };

  saveYAMLConfig(config);
}

/**
 * Add a capability to config
 */
export function addCapabilityYAML(
  name: string,
  capConfig: CapabilityConfig
): void {
  const config = loadYAMLConfig();

  if (config.capabilities[name]) {
    throw new Error(`Capability "${name}" already exists`);
  }

  if (!config.services[capConfig.service]) {
    throw new Error(`Service "${capConfig.service}" not found`);
  }

  config.capabilities[name] = capConfig;
  saveYAMLConfig(config);
}

/**
 * Migrate from JSON to YAML config
 */
export function migrateToYAML(): void {
  if (!fs.existsSync(CONFIG_FILE_JSON)) {
    throw new Error('No JSON config to migrate');
  }

  if (fs.existsSync(CONFIG_FILE_YAML)) {
    throw new Error('YAML config already exists');
  }

  // Load old JSON config
  const oldConfig = JSON.parse(fs.readFileSync(CONFIG_FILE_JSON, 'utf8'));

  // Create new YAML config
  const newConfig: JaneeYAMLConfig = {
    version: '0.2.0',
    masterKey: oldConfig.masterKey,
    server: {
      port: oldConfig.settings?.port || 9119,
      host: 'localhost'
    },
    services: {},
    capabilities: {}
  };

  // Migrate services
  if (oldConfig.services) {
    for (const service of oldConfig.services) {
      newConfig.services[service.name] = {
        baseUrl: service.baseUrl,
        auth: {
          type: 'bearer',
          key: service.encryptedKey  // Already encrypted
        }
      };

      // Create default capability for each service
      newConfig.capabilities[service.name] = {
        service: service.name,
        ttl: '1h',
        autoApprove: true
      };
    }
  }

  // Save (will re-encrypt with YAML format)
  const yamlContent = yaml.dump(newConfig, {
    indent: 2,
    lineWidth: 120
  });

  fs.writeFileSync(CONFIG_FILE_YAML, yamlContent, { mode: 0o600 });

  console.log('✅ Migrated to YAML config');
  console.log(`Old config backed up at: ${CONFIG_FILE_JSON}.bak`);
  
  // Backup old config
  fs.renameSync(CONFIG_FILE_JSON, `${CONFIG_FILE_JSON}.bak`);
}
