import { getConfigDir } from '../config-yaml';
import fs from 'fs';
import path from 'path';

interface Session {
  id: string;
  capability: string;
  service: string;
  agentId?: string;
  reason?: string;
  createdAt: string;
  expiresAt: string;
  revoked: boolean;
}

export async function sessionsCommand(): Promise<void> {
  try {
    const sessionsFile = path.join(getConfigDir(), 'sessions.json');

    if (!fs.existsSync(sessionsFile)) {
      console.log('No active sessions');
      console.log('');
      console.log('Sessions will appear when agents access APIs via MCP.');
      return;
    }

    const data = fs.readFileSync(sessionsFile, 'utf8');
    const sessions: Session[] = JSON.parse(data);

    // Filter active sessions
    const now = new Date();
    const active = sessions.filter(s => {
      return !s.revoked && new Date(s.expiresAt) > now;
    });

    if (active.length === 0) {
      console.log('No active sessions');
      return;
    }

    console.log('');
    console.log('Active sessions:');
    console.log('');

    active.forEach(session => {
      const expires = new Date(session.expiresAt);
      const ttl = Math.floor((expires.getTime() - now.getTime()) / 1000);
      const ttlStr = formatTTL(ttl);

      console.log(`  ${session.id.substring(0, 20)}...`);
      console.log(`    Capability: ${session.capability}`);
      console.log(`    Service: ${session.service}`);
      if (session.agentId) {
        console.log(`    Agent: ${session.agentId}`);
      }
      if (session.reason) {
        console.log(`    Reason: ${session.reason}`);
      }
      console.log(`    Expires: ${ttlStr}`);
      console.log('');
    });

    console.log(`Total: ${active.length} active session${active.length === 1 ? '' : 's'}`);
    console.log('');

  } catch (error) {
    if (error instanceof Error) {
      console.error('❌ Error:', error.message);
    } else {
      console.error('❌ Unknown error occurred');
    }
    process.exit(1);
  }
}

function formatTTL(seconds: number): string {
  if (seconds < 60) return `${seconds}s`;
  if (seconds < 3600) return `${Math.floor(seconds / 60)}m`;
  if (seconds < 86400) return `${Math.floor(seconds / 3600)}h`;
  return `${Math.floor(seconds / 86400)}d`;
}
