import { getConfigDir } from '../config-yaml';
import fs from 'fs';
import path from 'path';

interface Session {
  id: string;
  capability: string;
  service: string;
  agentId?: string;
  reason?: string;
  createdAt: string;
  expiresAt: string;
  revoked: boolean;
}

export async function revokeCommand(sessionIdPrefix: string): Promise<void> {
  try {
    const sessionsFile = path.join(getConfigDir(), 'sessions.json');

    if (!fs.existsSync(sessionsFile)) {
      console.error('❌ No sessions file found');
      process.exit(1);
    }

    const data = fs.readFileSync(sessionsFile, 'utf8');
    const sessions: Session[] = JSON.parse(data);

    // Find session by prefix
    const session = sessions.find(s => s.id.startsWith(sessionIdPrefix));

    if (!session) {
      console.error(`❌ Session not found: ${sessionIdPrefix}`);
      console.error('');
      console.error('Run: janee sessions');
      process.exit(1);
    }

    if (session.revoked) {
      console.log(`⚠️  Session already revoked: ${session.id.substring(0, 20)}...`);
      return;
    }

    // Revoke session
    session.revoked = true;

    // Save back
    fs.writeFileSync(sessionsFile, JSON.stringify(sessions, null, 2), { mode: 0o600 });

    console.log(`✅ Session revoked: ${session.id.substring(0, 20)}...`);
    console.log('');
    console.log(`   Capability: ${session.capability}`);
    console.log(`   Service: ${session.service}`);
    if (session.agentId) {
      console.log(`   Agent: ${session.agentId}`);
    }
    console.log('');
    console.log('Agent will lose access immediately on next request.');

  } catch (error) {
    if (error instanceof Error) {
      console.error('❌ Error:', error.message);
    } else {
      console.error('❌ Unknown error occurred');
    }
    process.exit(1);
  }
}
