import * as readline from 'readline/promises';
import { stdin as input, stdout as output } from 'process';
import { loadYAMLConfig, saveYAMLConfig, hasYAMLConfig } from '../config-yaml';

export async function removeCommand(serviceName: string): Promise<void> {
  try {
    // Check for YAML config
    if (!hasYAMLConfig()) {
      console.error('❌ No config found. Run `janee init` first.');
      process.exit(1);
    }

    const config = loadYAMLConfig();

    // Check if service exists
    if (!config.services[serviceName]) {
      console.error(`❌ Service "${serviceName}" not found`);
      process.exit(1);
    }

    const rl = readline.createInterface({ input, output });

    // Check for capabilities using this service
    const dependentCaps = Object.entries(config.capabilities)
      .filter(([_, cap]) => cap.service === serviceName)
      .map(([name, _]) => name);

    if (dependentCaps.length > 0) {
      console.log(`⚠️  The following capabilities depend on this service:`);
      dependentCaps.forEach(cap => console.log(`   - ${cap}`));
      console.log();
    }

    // Confirm deletion
    const answer = await rl.question(
      `Are you sure you want to remove service "${serviceName}"? This cannot be undone. (y/N): `
    );

    rl.close();

    if (answer.toLowerCase() !== 'y' && answer.toLowerCase() !== 'yes') {
      console.log('❌ Cancelled');
      return;
    }

    // Remove service
    delete config.services[serviceName];

    // Remove dependent capabilities
    dependentCaps.forEach(cap => {
      delete config.capabilities[cap];
    });

    saveYAMLConfig(config);

    console.log(`✅ Service "${serviceName}" removed successfully!`);

    if (dependentCaps.length > 0) {
      console.log(`✅ Removed ${dependentCaps.length} dependent capability(ies)`);
    }

  } catch (error) {
    if (error instanceof Error) {
      console.error('❌ Error:', error.message);
    } else {
      console.error('❌ Unknown error occurred');
    }
    process.exit(1);
  }
}
