import { AuditLogger } from '../../core/audit';
import { getAuditDir } from '../config-yaml';

export async function logsCommand(options: {
  follow?: boolean;
  lines?: string;
  service?: string;
}): Promise<void> {
  try {
    const auditLogger = new AuditLogger(getAuditDir());

    if (options.follow) {
      // Tail logs in real-time
      console.log('Following logs (Ctrl+C to stop)...\n');

      for await (const event of auditLogger.tail()) {
        const timestamp = new Date(event.timestamp).toLocaleTimeString();
        const statusColor = event.statusCode && event.statusCode >= 400 ? '\x1b[31m' : '\x1b[32m';
        const reset = '\x1b[0m';
        
        console.log(
          `${timestamp} ${event.method.padEnd(6)} /${event.service}${event.path} ${statusColor}${event.statusCode || '---'}${reset}`
        );
      }
    } else {
      // Show recent logs
      const limit = parseInt(options.lines || '20');
      const events = await auditLogger.readLogs({
        limit,
        service: options.service
      });

      if (events.length === 0) {
        console.log('No logs found.');
        console.log('');
        console.log('Logs will appear when you use the proxy:');
        console.log('  janee serve');
        return;
      }

      console.log('');
      console.log(`Recent activity (last ${events.length} requests):`);
      console.log('');

      events.reverse().forEach(event => {
        const timestamp = new Date(event.timestamp).toLocaleString();
        const statusColor = event.statusCode && event.statusCode >= 400 ? '\x1b[31m' : '\x1b[32m';
        const reset = '\x1b[0m';
        
        console.log(
          `${timestamp} ${event.method.padEnd(6)} /${event.service}${event.path} ${statusColor}${event.statusCode || '---'}${reset}`
        );
      });

      console.log('');
    }

  } catch (error) {
    if (error instanceof Error) {
      console.error('❌ Error:', error.message);
    } else {
      console.error('❌ Unknown error occurred');
    }
    process.exit(1);
  }
}
