import fs from 'fs';
import path from 'path';
import os from 'os';
import { generateMasterKey } from '../../core/crypto';

const CONFIG_DIR = path.join(os.homedir(), '.janee');
const CONFIG_FILE = path.join(CONFIG_DIR, 'config.yaml');
const LOGS_DIR = path.join(CONFIG_DIR, 'logs');

const EXAMPLE_CONFIG = `# Janee Configuration
# Edit this file to add your API services and configure access policies

version: '0.2.0'
masterKey: '{{MASTER_KEY}}'

# LLM for adjudication (optional - Phase 2 feature)
# llm:
#   provider: openai  # or anthropic
#   apiKey: env:OPENAI_API_KEY
#   model: gpt-4o-mini

# Your API services - add your real services here
services:
  # Example: Stripe API
  # stripe:
  #   baseUrl: https://api.stripe.com
  #   auth:
  #     type: bearer
  #     key: sk_live_xxx  # Will be encrypted at rest

  # Example: GitHub API
  # github:
  #   baseUrl: https://api.github.com
  #   auth:
  #     type: bearer
  #     key: ghp_xxx

  # Example: Exchange API with HMAC auth
  # bybit:
  #   baseUrl: https://api.bybit.com
  #   auth:
  #     type: hmac
  #     apiKey: xxx
  #     apiSecret: xxx

  # Example: API with custom headers
  # custom:
  #   baseUrl: https://api.example.com
  #   auth:
  #     type: headers
  #     headers:
  #       X-API-Key: xxx
  #       X-Custom-Header: yyy

# Capabilities - what agents can access
capabilities:
  # Example: Read-only Stripe access (with path policies)
  # stripe_readonly:
  #   service: stripe
  #   ttl: 1h
  #   autoApprove: true
  #   rules:
  #     allow:
  #       - GET *
  #     deny:
  #       - POST *
  #       - PUT *
  #       - DELETE *

  # Example: Stripe billing operations (limited write access)
  # stripe_billing:
  #   service: stripe
  #   ttl: 15m
  #   requiresReason: true
  #   rules:
  #     allow:
  #       - GET *
  #       - POST /v1/refunds/*
  #       - POST /v1/invoices/*
  #     deny:
  #       - POST /v1/charges/*
  #       - DELETE *

  # Example: GitHub access (no restrictions)
  # github:
  #   service: github
  #   ttl: 30m
  #   autoApprove: true
`;

export async function initCommand(): Promise<void> {
  try {
    // Ensure config directory exists
    if (!fs.existsSync(CONFIG_DIR)) {
      fs.mkdirSync(CONFIG_DIR, { mode: 0o700, recursive: true });
    }

    // Ensure logs directory exists
    if (!fs.existsSync(LOGS_DIR)) {
      fs.mkdirSync(LOGS_DIR, { mode: 0o700, recursive: true });
    }

    // Check if config already exists
    if (fs.existsSync(CONFIG_FILE)) {
      console.error('❌ Config already exists at:', CONFIG_FILE);
      console.error('');
      console.error('To start fresh, remove the existing config:');
      console.error(`  rm ${CONFIG_FILE}`);
      process.exit(1);
    }

    // Generate master key
    const masterKey = generateMasterKey();

    // Create config from template
    const config = EXAMPLE_CONFIG.replace('{{MASTER_KEY}}', masterKey);

    // Write config file
    fs.writeFileSync(CONFIG_FILE, config, { mode: 0o600 });

    console.log('✅ Janee initialized successfully!');
    console.log();
    console.log(`Config file: ${CONFIG_FILE}`);
    console.log(`Logs directory: ${LOGS_DIR}`);
    console.log();
    console.log('Next steps:');
    console.log(`  1. Edit ${CONFIG_FILE}`);
    console.log('     Uncomment the example services and add your API keys');
    console.log('  2. Start the MCP server:  janee serve');
    console.log('  3. Connect your agent (OpenClaw, Claude Desktop, etc.)');
    console.log();

  } catch (error) {
    if (error instanceof Error) {
      console.error('❌ Error:', error.message);
    } else {
      console.error('❌ Unknown error occurred');
    }
    process.exit(1);
  }
}
