import * as readline from 'readline/promises';
import { stdin as input, stdout as output } from 'process';
import { loadYAMLConfig, saveYAMLConfig, hasYAMLConfig } from '../config-yaml';
import type { AuthConfig, ServiceConfig, CapabilityConfig } from '../config-yaml';

export async function addCommand(
  serviceName?: string,
  options: { url?: string; key?: string; description?: string } = {}
): Promise<void> {
  try {
    // Check for YAML config
    if (!hasYAMLConfig()) {
      console.error('❌ No config found. Run `janee init` first.');
      process.exit(1);
    }

    const rl = readline.createInterface({ input, output });

    // Service name
    if (!serviceName) {
      serviceName = await rl.question('Service name: ');
      serviceName = serviceName.trim();
    }

    if (!serviceName) {
      console.error('❌ Service name is required');
      rl.close();
      process.exit(1);
    }

    const config = loadYAMLConfig();

    // Check if service already exists
    if (config.services[serviceName]) {
      console.error(`❌ Service "${serviceName}" already exists`);
      rl.close();
      process.exit(1);
    }

    // Base URL
    let baseUrl = options.url;
    if (!baseUrl) {
      baseUrl = await rl.question('Base URL: ');
      baseUrl = baseUrl.trim();
    }

    if (!baseUrl || !baseUrl.startsWith('http')) {
      console.error('❌ Invalid base URL. Must start with http:// or https://');
      rl.close();
      process.exit(1);
    }

    // Auth type
    const authTypeInput = await rl.question('Auth type (bearer/hmac/headers): ');
    const authType = authTypeInput.trim().toLowerCase() as 'bearer' | 'hmac' | 'headers';

    if (!['bearer', 'hmac', 'headers'].includes(authType)) {
      console.error('❌ Invalid auth type. Must be bearer, hmac, or headers');
      rl.close();
      process.exit(1);
    }

    // Build auth config
    let auth: AuthConfig;

    if (authType === 'bearer') {
      let apiKey = options.key;
      if (!apiKey) {
        apiKey = await rl.question('API key: ');
        apiKey = apiKey.trim();
      }

      if (!apiKey) {
        console.error('❌ API key is required');
        rl.close();
        process.exit(1);
      }

      auth = {
        type: 'bearer',
        key: apiKey
      };
    } else if (authType === 'hmac') {
      const apiKey = await rl.question('API key: ');
      const apiSecret = await rl.question('API secret: ');

      if (!apiKey || !apiSecret) {
        console.error('❌ API key and secret are required for HMAC');
        rl.close();
        process.exit(1);
      }

      auth = {
        type: 'hmac',
        apiKey: apiKey.trim(),
        apiSecret: apiSecret.trim()
      };
    } else {
      // headers
      console.log('Enter headers as key:value pairs (empty line to finish):');
      const headers: Record<string, string> = {};

      while (true) {
        const line = await rl.question('  ');
        if (!line.trim()) break;

        const [key, ...valueParts] = line.split(':');
        const value = valueParts.join(':').trim();

        if (key && value) {
          headers[key.trim()] = value;
        }
      }

      if (Object.keys(headers).length === 0) {
        console.error('❌ At least one header is required');
        rl.close();
        process.exit(1);
      }

      auth = {
        type: 'headers',
        headers
      };
    }

    // Add service to config
    config.services[serviceName] = {
      baseUrl,
      auth
    };

    saveYAMLConfig(config);

    console.log(`✅ Added service "${serviceName}"`);
    console.log();

    // Ask about capability
    const createCapAnswer = await rl.question('Create a capability for this service? (Y/n): ');
    const createCap = !createCapAnswer || createCapAnswer.toLowerCase() === 'y' || createCapAnswer.toLowerCase() === 'yes';

    if (createCap) {
      const capNameDefault = serviceName;
      const capNameInput = await rl.question(`Capability name (default: ${capNameDefault}): `);
      const capName = capNameInput.trim() || capNameDefault;

      // Check if capability already exists
      if (config.capabilities[capName]) {
        console.error(`❌ Capability "${capName}" already exists`);
        rl.close();
        process.exit(1);
      }

      const ttlInput = await rl.question('TTL (e.g., 1h, 30m): ');
      const ttl = ttlInput.trim() || '1h';

      const autoApproveInput = await rl.question('Auto-approve? (Y/n): ');
      const autoApprove = !autoApproveInput || autoApproveInput.toLowerCase() === 'y' || autoApproveInput.toLowerCase() === 'yes';

      const requiresReasonInput = await rl.question('Requires reason? (y/N): ');
      const requiresReason = requiresReasonInput.toLowerCase() === 'y' || requiresReasonInput.toLowerCase() === 'yes';

      // Add capability
      config.capabilities[capName] = {
        service: serviceName,
        ttl,
        autoApprove,
        requiresReason
      };

      saveYAMLConfig(config);

      console.log(`✅ Added capability "${capName}"`);
      console.log();
    }

    rl.close();

    console.log("Done! Run 'janee serve' to start.");

  } catch (error) {
    if (error instanceof Error) {
      console.error('❌ Error:', error.message);
    } else {
      console.error('❌ Unknown error occurred');
    }
    process.exit(1);
  }
}
