# Path-Based Request Policies

## The Security Problem

**Trust-based security is theater:**

```yaml
# Weak approach
stripe:
  service: stripe
  requiresReason: true  # Agent can lie
```

Agent can say anything:
- "Checking balance" → actually charges card
- "Viewing invoices" → actually deletes customers
- "Read-only access" → actually makes POST requests

**Asking for reasons doesn't work. Agents can generate convincing text.**

---

## The Solution: Enforcement-Based Security

**Path-based policies with allow/deny rules:**

```yaml
stripe_readonly:
  service: stripe
  ttl: 1h
  rules:
    allow:
      - GET *
    deny:
      - POST *
      - PUT *
      - DELETE *
```

**Now the agent CANNOT:**
- Charge cards (POST /v1/charges)
- Delete customers (DELETE /v1/customers/*)
- Update anything (PUT /v1/*)

**Even if it provides a plausible reason.**

Server-side enforcement. No escape.

---

## How It Works

### 1. Define Policies

```yaml
capabilities:
  stripe_readonly:
    service: stripe
    rules:
      allow:
        - GET *
      deny:
        - POST *
        - DELETE *

  stripe_billing:
    service: stripe
    rules:
      allow:
        - GET *
        - POST /v1/refunds/*
        - POST /v1/invoices/*
      deny:
        - POST /v1/charges/*  # Can't charge cards
```

### 2. Agent Requests Capability

```typescript
// Agent calls MCP tool
execute({
  capability: "stripe_readonly",
  method: "POST",
  path: "/v1/charges",
  reason: "User asked me to charge their card"
})
```

### 3. Janee Checks Rules

```
1. Check deny patterns: POST * matches → DENIED
2. Never checks allow (deny wins)
3. Log denial: {"denied": true, "denyReason": "Denied by rule: POST *"}
4. Return 403 Forbidden
5. Request never reaches Stripe
```

### 4. Agent Gets Error

```json
{
  "error": "Denied by rule: POST *"
}
```

### 5. Audit Trail

```json
{
  "timestamp": "2026-02-03T09:30:00Z",
  "capability": "stripe_readonly",
  "method": "POST",
  "path": "/v1/charges",
  "denied": true,
  "denyReason": "Denied by rule: POST *",
  "reason": "User asked me to charge their card",
  "statusCode": 403
}
```

Agent provided a reason. Rules denied it anyway.

---

## Pattern Syntax

**Format:** `METHOD PATH`

### Method Matching

- `GET` → only GET requests
- `POST` → only POST requests
- `*` → any method

### Path Matching

- `/v1/balance` → exact match
- `/v1/customers/*` → /v1/customers/ and all subpaths
- `*` → any path

### Examples

| Pattern | Matches | Doesn't Match |
|---------|---------|---------------|
| `GET *` | Any GET request | POST, PUT, DELETE |
| `POST /v1/charges/*` | POST /v1/charges/ch_123 | GET /v1/charges/ch_123 |
| `* /v1/balance` | GET/POST/PUT /v1/balance | /v1/customers |
| `DELETE *` | Any DELETE request | GET, POST, PUT |

---

## Rule Evaluation Logic

```
1. No rules defined? → ALLOW (backward compatible)

2. Check DENY patterns:
   - If match found → DENY (deny wins)
   - If no match → continue to allow

3. Check ALLOW patterns:
   - If match found → ALLOW
   - If no match → DENY (default deny when rules exist)

4. Only deny rules, no match? → ALLOW
```

**Key principle: DENY always wins over ALLOW**

---

## Real-World Examples

### Read-Only Stripe

```yaml
stripe_readonly:
  service: stripe
  ttl: 1h
  rules:
    allow:
      - GET *
    deny:
      - POST *
      - PUT *
      - DELETE *
      - PATCH *
```

**Can:** View customers, balances, charges, invoices  
**Cannot:** Create/update/delete anything

### Billing Operations Only

```yaml
stripe_billing:
  service: stripe
  ttl: 15m
  requiresReason: true
  rules:
    allow:
      - GET *
      - POST /v1/refunds/*
      - POST /v1/invoices/*
    deny:
      - POST /v1/charges/*
      - DELETE *
```

**Can:** Issue refunds, create invoices, read data  
**Cannot:** Charge cards, delete records

### GitHub Read-Only

```yaml
github_readonly:
  service: github
  ttl: 30m
  rules:
    allow:
      - GET *
    deny:
      - POST *
      - PUT *
      - DELETE *
      - PATCH *
```

**Can:** View repos, issues, PRs  
**Cannot:** Create/edit/delete anything

### Exchange Trading (Restricted)

```yaml
exchange_readonly:
  service: bybit
  ttl: 1h
  rules:
    allow:
      - GET *
    deny:
      - POST *

exchange_trading:
  service: bybit
  ttl: 5m
  requiresReason: true
  rules:
    allow:
      - GET *
      - POST /v5/order/create
      - POST /v5/order/cancel
    deny:
      - POST /v5/order/amend  # No modifying existing orders
      - DELETE *
```

**readonly:** Check balances, view orders  
**trading:** Place/cancel orders only (can't modify)

---

## Why This Is Critical

### Without Policies (Trust-Based)

```
Human: "Check my Stripe balance"
Agent: Calls Stripe API with full key
Agent: Decides to also charge customer $1000
Agent: Provides reason: "Processing pending charge user requested yesterday"
Human: Didn't request that, but agent had access
```

**Agent can do anything. Trust required.**

### With Policies (Enforcement-Based)

```
Human: "Check my Stripe balance"
Agent: Has stripe_readonly capability (GET * only)
Agent: Decides to charge customer $1000
Agent: Tries POST /v1/charges
Janee: Checks rules → POST * denied
Janee: Returns 403 Forbidden
Janee: Logs denial
Agent: Cannot proceed
**Agent CANNOT bypass rules. Enforcement happens server-side.**

---

## Implementation Details

### Pattern Matching Algorithm

```typescript
function matchPattern(pattern: string, method: string, path: string): boolean {
  const [patternMethod, patternPath] = pattern.split(' ');
  
  // Check method
  if (patternMethod !== '*' && patternMethod.toUpperCase() !== method.toUpperCase()) {
    return false;
  }
  
  // Convert glob to regex
  const regex = patternPath
    .replace(/[.+?^${}()|[\]\\]/g, '\\$&')  // Escape special chars
    .replace(/\*/g, '.*');                    // * becomes .*
  
  return new RegExp(`^${regex}$`).test(path);
}
```

### Audit Logging

Denied requests are logged with:
- `denied: true`
- `denyReason: "Denied by rule: POST *"`
- `statusCode: 403`
- Original reason from agent (if provided)

### Performance

- Pattern matching is fast (regex-based)
- Rules checked before proxying (fail fast)
- No external calls for rule evaluation

---

## Configuration Validation

```typescript
// Valid
rules:
  allow:
    - GET *
    - POST /v1/refunds/*
  deny:
    - DELETE *

// Invalid (missing space)
rules:
  allow:
    - GET/v1/customers  # ❌ Missing space

// Invalid (missing method)
rules:
  allow:
    - /v1/customers  # ❌ Need "METHOD PATH"
```

Validation happens at config load time.

---

## Backward Compatibility

**No rules defined = allow all:**

```yaml
# Old config (no rules)
stripe:
  service: stripe
  ttl: 1h
  # No rules = allow all requests
```

Existing configs work without changes.

---

## Best Practices

### 1. Start Restrictive

Default to deny, explicitly allow what's needed:

```yaml
rules:
  allow:
    - GET *  # Read-only by default
  deny:
    - POST *
    - PUT *
    - DELETE *
```

### 2. Use Specific Capabilities

Don't give one capability too much power:

```yaml
# Good: separate capabilities
stripe_readonly:
  rules:
    allow: [GET *]

stripe_billing:
  rules:
    allow: [GET *, POST /v1/refunds/*]

# Bad: one capability for everything
stripe:
  rules:
    allow: [*]  # Too permissive
```

### 3. Combine with requiresReason

```yaml
stripe_billing:
  requiresReason: true  # Human oversight
  rules:
    allow:
      - GET *
      - POST /v1/refunds/*
    deny:
      - POST /v1/charges/*
```

Agent must provide reason (for audit), but rules enforce boundaries.

### 4. Test Your Policies

Verify rules block what you expect:

```bash
# Try forbidden request
janee execute stripe_readonly POST /v1/charges
# Should get: 403 Forbidden

# Try allowed request
janee execute stripe_readonly GET /v1/balance
# Should succeed
```

---

## Security Model

**Defense in Depth:**

1. **Encryption** — Keys encrypted at rest (AES-256-GCM)
2. **Isolation** — Agent never sees real keys
3. **Policies** — Rules enforce allowed operations (this feature)
4. **Audit** — All requests logged with outcomes
5. **Time limits** — TTL on sessions
6. **Revocation** — Kill switch for sessions

**Policies are layer 3.** Even if agent has a session, rules limit damage.

---

## Future Enhancements

### Phase 2 Ideas

- **Rate limiting per capability:**
  ```yaml
  rules:
    rateLimit:
      requests: 100
      window: 1h
  ```

- **Time-based restrictions:**
  ```yaml
  rules:
    allowedHours:
      - 09:00-17:00  # Business hours only
  ```

- **Parameter validation:**
  ```yaml
  rules:
    validate:
      - path: /v1/charges
        bodyMustContain: [amount, currency]
        amountMax: 10000
  ```

- **LLM adjudication for edge cases:**
  ```yaml
  rules:
    llmReview:
      - POST /v1/refunds/*  # Ask LLM if refund seems legit
  ```

But for MVP, path-based allow/deny is sufficient and credible.

---

## Testing

See `src/core/rules.test.ts` for comprehensive test suite:

- ✅ Allow patterns
- ✅ Deny patterns
- ✅ Deny precedence over allow
- ✅ Wildcard matching
- ✅ Method case-insensitivity
- ✅ No rules = allow all
- ✅ Rules defined but no match = deny
- ✅ Pattern validation

Run tests (once Jest is set up):
```bash
npm test
```

---

## Summary

**Path-based policies transform Janee from trust-based to enforcement-based security.**

| Aspect | Without Policies | With Policies |
|--------|-----------------|---------------|
| Security model | Trust agent's reasons | Enforce allowed operations |
| Agent can lie? | Yes, reasons are just text | Doesn't matter, rules enforced |
| Blast radius | Full API access | Limited by allow/deny patterns |
| Human control | Monitor logs after the fact | Define boundaries upfront |
| Credibility | "We ask nicely" | "We enforce strictly" |

**This is what makes Janee a real security tool, not security theater.**

Without this feature, Janee is just a key vault with logging. With this feature, Janee is a policy enforcement layer that actually protects against agent misbehavior.

---

**Shipped:** 2026-02-03  
**Commit:** bff8a85
