---
name: jack-cloud
description: >
  Deploy web services to the cloud with Jack.
  Use when: you need to create APIs, websites, or backends and deploy them live.
  Teaches: project creation, deployment, databases, logs, and all Jack Cloud services.
homepage: https://getjack.org
metadata: {"clawdbot":{"emoji":"🃏","homepage":"https://github.com/getjack-org/skills","requires":{"bins":["node","npm"],"env":[]},"install":[{"id":"npm","kind":"npm","package":"@getjack/jack","bins":["jack"],"label":"Install Jack CLI (npm)"}]}}
allowed-tools: Read, Edit, Grep, Glob
---

# Jack Cloud — Deploy Anything from the Terminal

Jack deploys Cloudflare Workers projects in one command. Create an API, add a database, ship it live — all from the terminal.

## Install

```bash
npm i -g @getjack/jack
jack login
```

## External Endpoints

| Endpoint | Data Sent | Purpose |
|----------|-----------|---------|
| `auth.getjack.org` | OAuth tokens (GitHub/Google via WorkOS) | Authentication |
| `control.getjack.org` | Project metadata, source code during deploy | Project management and deployments |

## Security & Privacy

- `jack login` authenticates via browser OAuth (GitHub/Google via WorkOS). Auth token stored at `~/.config/jack/auth.json`
- No environment variables required — authentication is interactive
- Source code is uploaded during `jack ship` and deployed to Cloudflare Workers via Jack Cloud
- Project metadata (name, slug, deploy history) is stored on Jack Cloud
- No telemetry is sent without user consent (`jack telemetry` to configure)
- **npm package:** [@getjack/jack](https://www.npmjs.com/package/@getjack/jack) — open source CLI

## MCP Tools

If your agent has `mcp__jack__*` tools available, prefer those over CLI commands. They return structured JSON and are tracked automatically. The CLI equivalents are noted below for agents without MCP.

---

## Create & Deploy a Project

```bash
jack new my-api
```

This creates a project from a template, deploys it, and prints the live URL.

**Pick a template** when prompted (or pass `--template`):

| Template | What you get |
|----------|-------------|
| `api` | Hono API with example routes |
| `hello` | Minimal hello-world starter |
| `miniapp` | Full-stack app with frontend |
| `ai-chat` | AI chat app with streaming |
| `nextjs` | Next.js full-stack app |

Run `jack new` to see all available templates.

**MCP:** `mcp__jack__create_project` with `name` and `template` params.

After creation, your project is live at `https://<slug>.runjack.xyz`.

---

## Deploy Changes

After editing code, push changes live:

```bash
jack ship
```

For machine-readable output (useful in scripts and agents):

```bash
jack ship --json
```

Builds the project and deploys to production. Takes a few seconds.

**MCP:** `mcp__jack__deploy_project`

---

## Check Status

```bash
jack info
```

Shows: live URL, last deploy time, attached services (databases, storage, etc.).

**MCP:** `mcp__jack__get_project_status`

---

## Database (D1)

```bash
jack services db create                  # Add D1 database (auto-configures wrangler.jsonc)
jack db execute "SELECT * FROM users"    # Query data
jack db execute --json "SELECT ..."      # JSON output
jack db execute --write "INSERT INTO users (name, email) VALUES ('Alice', 'alice@example.com')"
jack db execute --write "CREATE TABLE posts (id INTEGER PRIMARY KEY, title TEXT, body TEXT, created_at TEXT DEFAULT CURRENT_TIMESTAMP)"
jack db execute "SELECT name FROM sqlite_master WHERE type='table'"   # View schema
jack db execute "PRAGMA table_info(users)"
```

After schema changes, redeploy with `jack ship`.

**MCP:** `mcp__jack__create_database`, `mcp__jack__execute_sql` (set `allow_write: true` for writes; DROP/TRUNCATE blocked by default).

---

## Logs

Stream production logs to debug issues:

```bash
jack logs
```

Shows real-time request/response logs. Press Ctrl+C to stop.

**MCP:** `mcp__jack__tail_logs` with `duration_ms` and `max_events` params for a bounded sample.

---

## Common Workflow: API with Database

```bash
# 1. Create project
jack new my-api --template api

# 2. Add database
jack services db create

# 3. Create tables
jack db execute --write "CREATE TABLE items (id INTEGER PRIMARY KEY, name TEXT, created_at TEXT DEFAULT CURRENT_TIMESTAMP)"

# 4. Edit src/index.ts — add routes that query the DB
#    Access DB via: c.env.DB (the D1 binding)

# 5. Deploy
jack ship

# 6. Verify
curl https://my-api.runjack.xyz/api/items
```

---

## Secrets

Store API keys and sensitive values:

```bash
# Set a secret (prompts for value)
jack secrets set STRIPE_SECRET_KEY

# Set multiple
jack secrets set API_KEY WEBHOOK_SECRET

# List secrets (names only, values hidden)
jack secrets list
```

Secrets are available in your worker as `c.env.SECRET_NAME`. Redeploy after adding secrets:

```bash
jack ship
```

---

## Project Structure

```
my-project/
├── src/
│   └── index.ts          # Worker entry point
├── wrangler.jsonc        # Config: bindings, routes, compatibility
├── package.json
└── .jack/
    └── project.json      # Links to Jack Cloud
```

- `wrangler.jsonc` defines D1 bindings, environment vars, compatibility flags
- `.jack/project.json` links the local directory to your Jack Cloud project
- `src/index.ts` is the main entry point — typically a Hono app

---

## Advanced Services

### Storage (R2)

```bash
jack services storage create          # Create R2 bucket
jack services storage list            # List buckets
jack services storage info            # Bucket details
```

Access in worker via `c.env.BUCKET` binding. Use for file uploads, images, assets.

**MCP:** `mcp__jack__create_storage_bucket`, `mcp__jack__list_storage_buckets`, `mcp__jack__get_storage_info`

### Vector Search (Vectorize)

```bash
jack services vectorize create                    # Create index (768 dims, cosine)
jack services vectorize create --dimensions 1536  # Custom dimensions
jack services vectorize list
jack services vectorize info
```

Access via `c.env.VECTORIZE_INDEX` binding. Use for semantic search, RAG, embeddings.

**MCP:** `mcp__jack__create_vectorize_index`, `mcp__jack__list_vectorize_indexes`, `mcp__jack__get_vectorize_info`

### Cron Scheduling

```bash
jack services cron create "*/15 * * * *"   # Every 15 minutes
jack services cron create "0 * * * *"      # Every hour
jack services cron list
jack services cron test "0 9 * * MON"      # Validate + show next runs
```

Your worker needs a `scheduled()` handler or `POST /__scheduled` route.

**MCP:** `mcp__jack__create_cron`, `mcp__jack__list_crons`, `mcp__jack__test_cron`

### Custom Domains

```bash
jack domain connect app.example.com      # Reserve domain
jack domain assign app.example.com       # Assign to current project
jack domain unassign app.example.com     # Unassign
jack domain disconnect app.example.com   # Fully remove
```

Follow the DNS instructions printed after `assign`. Typically add a CNAME record.

---

## List Projects

```bash
jack ls           # List all your projects
jack info my-api  # Details for a specific project
jack open my-api  # Open in browser
```

**MCP:** `mcp__jack__list_projects` with optional `filter` (all, local, deployed, cloud).

---

## Troubleshooting

| Problem | Fix |
|---------|-----|
| "Not authenticated" | Run `jack login` |
| "No wrangler config found" | Run from a jack project directory |
| "Database not found" | Run `jack services db create` |
| Deploy fails | Check `jack logs` for errors, fix code, `jack ship` again |
| Need to start over | `jack new` creates a fresh project |

---

## Reference

- [Services deep dive](reference/services-guide.md) — detailed patterns for each service
- [Jack documentation](https://docs.getjack.org)
