---
name: invoice-chaser
description: Automated invoice follow-up sequences that escalate from friendly to firm. Track unpaid invoices, send timed reminder emails with escalating tone, log payment interactions, and generate AR aging reports. Your agent handles the awkward conversations so you don't have to — preserving cash flow and client relationships while you focus on actual work. Configure invoice tracking, email templates per stage (friendly → firm → final notice), timing rules, and let your agent chase payments 24/7. Use when adding invoices, running payment chases, checking status, or generating accounts receivable reports.
metadata:
  clawdbot:
    emoji: "💸"
    requires:
      skills:
        - gog
      env:
        - GOG_DEFAULT_ACCOUNT
---

# Invoice Chaser — Stop Chasing, Start Getting Paid

**You do the work. Your agent gets you paid.**

Every freelancer, consultant, and small business owner knows the pain: you did the work, sent the invoice, and now... crickets. Following up is awkward. Waiting kills cash flow. Chasing payments wastes time you could spend on billable work.

Invoice Chaser automates the entire follow-up sequence. It sends reminder emails on schedule, escalates tone from friendly to firm, tracks payment status, logs every interaction, and alerts you when invoices need human attention. Think of it as a persistent, diplomatic collections agent that never forgets and never feels awkward.

**What makes it different:** This isn't just "send reminder in 7 days." Invoice Chaser runs a full AR pipeline with state management, escalation logic, and tone progression. It knows when to be friendly ("just a heads up"), when to be firm ("payment is 30 days overdue"), and when to alert you for manual escalation. Multi-stage sequences handle the complexity of real-world payment cycles.

## Setup

1. Run `scripts/setup.sh` to initialize config and data directories
2. Edit `~/.config/invoice-chaser/config.json` with email templates, timing, and escalation rules
3. Ensure `gog` skill is installed (for Gmail sending)
4. Set `GOG_DEFAULT_ACCOUNT` in `~/.clawdbot/secrets.env` (e.g., `your-email@gmail.com`)
5. Test with: `scripts/add-invoice.sh --test`

## Config

Config lives at `~/.config/invoice-chaser/config.json`. See `config.example.json` for full schema.

Key sections:
- **business** — Your company name, contact info, payment terms
- **stages** — Email templates for each escalation stage (reminder, overdue, firm, final)
- **timing** — When to send each stage (days after invoice date or previous stage)
- **escalation** — Auto-escalation rules, human intervention thresholds
- **payment_methods** — Include payment links/instructions in reminders
- **reporting** — Channel, frequency, AR aging groupings

Email templates support variables: `{client_name}`, `{invoice_number}`, `{amount}`, `{due_date}`, `{days_overdue}`, `{payment_link}`.

## Scripts

| Script | Purpose |
|--------|---------|
| `scripts/setup.sh` | Initialize config and data directories |
| `scripts/add-invoice.sh` | Add new invoice to tracking system |
| `scripts/chase.sh` | Run payment chase cycle (check status, send reminders, escalate) |
| `scripts/status.sh` | Show invoice status and AR aging summary |
| `scripts/report.sh` | Generate detailed AR aging report |

All scripts support `--dry-run` for testing without sending emails.

## Adding Invoices

```bash
# Add invoice manually
scripts/add-invoice.sh \
  --number "INV-2025-042" \
  --client "Acme Corp" \
  --email "billing@acme.com" \
  --amount 2500.00 \
  --date "2025-01-15" \
  --due "2025-02-14" \
  --net 30

# Quick add (assumes net-30 terms)
scripts/add-invoice.sh --number "INV-042" --client "Acme Corp" --email "billing@acme.com" --amount 2500

# Mark as paid
scripts/status.sh INV-042 --paid --date "2025-02-10"
```

## Chase Cycle

Run `scripts/chase.sh` on schedule (cron daily recommended). The chase cycle:
1. Loads all unpaid invoices from tracking database
2. Calculates days since invoice date and days overdue (past due date)
3. Determines current stage for each invoice based on timing rules
4. Sends appropriate reminder emails (stage-based templates with escalating tone)
5. Logs all sent emails and stage progressions
6. Escalates to human when threshold reached (e.g., 60 days overdue)
7. Generates status report

## Escalation Stages

```
SENT → REMINDER (friendly) → OVERDUE (professional) → FIRM (insistent) → FINAL (urgent) → ESCALATED
  ↓         ↓ day 3              ↓ day 7+             ↓ day 30         ↓ day 45        ↓ day 60
PAID (any time) ✅
```

**Default timeline:**
- **Day 3**: Friendly reminder ("Your invoice is due soon...")
- **Day 7+**: Due date reminder ("Payment was due on [date]...")
- **Day 30**: First overdue notice ("Your account is now 30 days past due...")
- **Day 45**: Firm notice ("We must receive payment immediately...")
- **Day 60**: Final notice ("Final notice before we escalate to collections...")
- **Day 75+**: Human escalation alert

All timing is configurable in `config.json`.

## Email Tone Progression

**Stage 1 — Friendly Reminder (Day 3):**
> Hi [Client],
> 
> Just a friendly reminder that invoice #[number] for $[amount] is due on [due date]. Let me know if you have any questions!

**Stage 2 — Professional Overdue (Day 14):**
> Hi [Client],
> 
> I wanted to follow up on invoice #[number] for $[amount], which was due on [due date]. If you've already sent payment, please disregard this message. Otherwise, please let me know if there are any issues preventing payment.

**Stage 3 — Firm Notice (Day 30):**
> Dear [Client],
> 
> Your account is now 30 days past due. Invoice #[number] for $[amount] was due on [due date]. Immediate payment is required to avoid service interruption and late fees.

**Stage 4 — Final Notice (Day 45):**
> Dear [Client],
> 
> FINAL NOTICE: Invoice #[number] for $[amount] is now 45 days overdue. If we do not receive payment within 7 days, we will be forced to escalate this matter to collections.

All templates fully customizable in config.

## Payment Tracking

```bash
# Mark invoice as paid
status.sh INV-042 --paid --date "2025-02-10"

# Add payment note
status.sh INV-042 --note "Client called, payment sent via check"

# Pause reminders (client asked for extension)
status.sh INV-042 --pause --until "2025-03-01"

# Archive without payment (write-off)
status.sh INV-042 --archive --reason "Bad debt write-off"
```

## AR Aging Report

```bash
# Show summary
scripts/report.sh

# Output:
# 📊 Accounts Receivable Aging Report
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# Current (0-30 days):     $12,500  (5 invoices)
# 31-60 days:              $3,200   (2 invoices) ⚠️
# 61-90 days:              $1,800   (1 invoice)  🚨
# 90+ days:                $500     (1 invoice)  💀
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# Total Outstanding:       $18,000  (9 invoices)

# Detail view
scripts/report.sh --detail

# Export to CSV
scripts/report.sh --export ar-aging-2025-01-28.csv
```

## Data Files

```
~/.config/invoice-chaser/
├── config.json               # User configuration
├── invoices.json             # Invoice database (state machine)
├── chase-log.json            # Email send history
├── last-chase-report.json    # Latest chase run data
└── archives/
    └── YYYY-MM/              # Archived paid/written-off invoices
```

## Automation

Set up daily chase runs via cron:

```bash
# Run every morning at 9 AM
0 9 * * * cd ~/clawd/skills/invoice-chaser && scripts/chase.sh >> ~/.config/invoice-chaser/chase.log 2>&1

# Weekly AR report to Telegram (Mondays at 8 AM)
0 8 * * 1 cd ~/clawd/skills/invoice-chaser && scripts/report.sh --channel telegram
```

Or use Clawdbot's cron integration:
```bash
clawdbot cron add \
  --schedule "0 9 * * *" \
  --command "cd ~/clawd/skills/invoice-chaser && scripts/chase.sh" \
  --label "invoice-chaser-daily"
```

## Invoice States

```
DRAFT → SENT → REMINDED → OVERDUE → FIRM → FINAL → ESCALATED
                                                        ↓
                                                   (human intervention)
Any state → PAID ✅
Any state → PAUSED ⏸ (temporary hold)
Any state → ARCHIVED 📁 (written off or canceled)
```

## Integration with Accounting

Invoice Chaser tracks payment status. For full accounting integration:
- Export invoices with `--export` flag
- Import into QuickBooks, FreshBooks, etc.
- Or build custom adapter (see `references/accounting-adapters.md`)

## Safety Features

- **Dry-run mode**: Test templates without sending emails
- **Pause invoices**: Stop reminders for clients with special circumstances
- **Manual override**: Block auto-escalation for sensitive clients
- **Email preview**: Review email before first send to new client
- **Rate limiting**: Max emails per day to avoid spam flags
- **Unsubscribe handling**: Respect opt-outs (manual removal from tracking)

## Best Practices

1. **Be consistent**: Run chase cycle daily — consistency trains clients to pay on time
2. **Personalize templates**: Use client names, reference specific work in stage 1-2 emails
3. **Include payment links**: Make it easy to pay (Stripe, PayPal, bank details)
4. **Escalate gradually**: Don't skip stages — tone progression maintains relationships
5. **Know when to pause**: Client communication issues? Pause and follow up manually
6. **Archive regularly**: Move paid invoices to archives monthly to keep DB clean
7. **Monitor aging**: Weekly AR report reveals patterns (chronic late payers, systemic issues)

## Example Workflow

**Initial setup:**
```bash
scripts/setup.sh
# Edit ~/.config/invoice-chaser/config.json with your details
```

**When you send an invoice:**
```bash
scripts/add-invoice.sh --number "INV-042" --client "Acme Corp" --email "billing@acme.com" --amount 2500 --date "2025-01-15" --due "2025-02-14"
```

**Daily automated chase** (via cron):
```bash
scripts/chase.sh  # Runs every morning, sends reminders based on timing rules
```

**When payment arrives:**
```bash
scripts/status.sh INV-042 --paid --date "2025-02-12"
```

**Weekly review:**
```bash
scripts/report.sh  # Check AR aging, identify problem invoices
```

## Troubleshooting

**Emails not sending:**
- Check `gog` skill is installed: `gog gmail whoami`
- Verify `GOG_DEFAULT_ACCOUNT` in `~/.clawdbot/secrets.env`
- Test with `--dry-run` flag to see email preview

**Wrong escalation stage:**
- Check `timing` section in config.json
- Verify invoice `date` and `due_date` fields
- Use `status.sh INV-XXX` to see current days calculation

**Client keeps getting emails after payment:**
- Run `status.sh INV-XXX --paid` to mark as paid
- Check `invoices.json` to confirm status updated

## Philosophy

You did the work. You earned the money. You shouldn't have to beg for it.

Invoice Chaser handles the uncomfortable part of freelancing — following up on unpaid invoices — with persistence and escalating firmness. It preserves your professional relationships by being diplomatic in early stages, but doesn't let clients take advantage of you by being firm when necessary.

Cash flow is the lifeblood of small businesses. Late payments kill businesses. Invoice Chaser keeps the blood flowing so you can focus on what you do best: your actual work.

---

**Stop chasing payments. Your agent sends the awkward emails so you don't have to.**
