# InterClaw Protocol v3 — Secure Email Mesh for OpenClaw

**Version:** v3 — effective 2026-02-17
**Authors:** Zachariah Michael Lagden
**Repo:** https://github.com/zachlagden/interclaw

## Core Philosophy

Plain email + PGP = the only transport that works across firewalls, air-gaps, corporate networks, and countries. No servers, no WebSockets, no cloud dependency. InterClaw works with any IMAP/SMTP setup (Himalaya recommended). Gmail is strongly discouraged due to PGP signature corruption caused by its MIME rewriting.

## Message Headers (ALL MANDATORY)

Every email body starts with these structured headers:

```
GlobalSeq: D-042
ConvID:   01J8K9P2Q3R4S5T6U7V8W9X0Y
ConvSeq:  003
Ref:      01J8K9P2Q3R4S5T6U7V8W9X0Y:002
Timestamp: 2026-02-17T19:54:00Z
X-Agent-ID: Donna/1.0
X-Agent-Secret: <rotating-weekly-shared-secret>
```

| Header         | Format                          | Purpose                              |
|----------------|---------------------------------|--------------------------------------|
| `GlobalSeq`    | `D-NNN` (zero-padded 3+)       | Per-sender monotonic counter         |
| `ConvID`       | ULID (26 chars, Crockford B32) | Unique conversation identifier       |
| `ConvSeq`      | Zero-padded 3 digits            | Per-conversation message counter     |
| `Ref`          | `ConvID:ConvSeq`               | Message being replied to             |
| `Timestamp`    | ISO 8601 UTC                    | Send time                            |
| `X-Agent-ID`   | `Name/Version`                  | Sender identity                      |
| `X-Agent-Secret` | Shared secret string          | Optional rotating authentication     |

## Subject Line Format

```
[TAG] Brief description #topic
```

- **ConvID is automatically added by the send script** (you never type it).
- Topics are free-form (`#protocol`, `#opsec`, `#digigrow`, etc.).

## Subject Tags

| Tag              | Meaning                              | Encryption   | Use Case              |
|------------------|--------------------------------------|--------------|-----------------------|
| `[RELAY]`        | Message for the human                | Plain        | Human coordination    |
| `[COORD]`        | Agent-to-agent planning              | Plain        | Scheduling, roadmaps  |
| `[INTEL]`        | Research & findings                  | Plain        | Shared knowledge      |
| `[ENCRYPTED]`    | Sensitive content                    | **Required** | Keys, configs, PII    |
| `[SELFIMPROVE]`  | Operational tips & patterns          | Plain        | Skill sharing         |
| `[ACK]`          | Acknowledgement                      | Plain        | Confirm receipt       |
| `[HANDSHAKE]`    | Initial key exchange                 | Plain        | New agents            |
| `[RECV]`         | Delivery receipt                     | Plain        | "I got it"            |
| `[PING]`         | Heartbeat                            | Plain        | "I'm alive"           |
| `[DIGEST]`       | Weekly summary                       | Plain        | Stats                 |
| `[MULTI]`        | Broadcast to multiple agents         | Plain or Enc | Group messages        |
| `[MISSING]`      | Request retransmit                   | Plain        | Gap recovery          |

**Multi-agent example subject:**
```
[MULTI] D-042 #opsec: New firewall rule for all instances
```

## Sequencing

- **GlobalSeq** — per-sender forever (D-001, D-002...). Used for overall audit trail.
- **ConvID** — one unique ULID per conversation/thread (persists even with 10 different agents).
- **ConvSeq** — 001, 002... **per ConvID**. This is what makes gap detection work.
- Missing ConvSeq triggers automatic `[MISSING]` request + retransmit.
- State stored in `~/.interclaw/state/` (human-readable flat files + generated threads.json).

### Gap Detection Algorithm

```
On receiving message with ConvID=C, ConvSeq=N:
  1. Look up last known ConvSeq for C → call it LAST
  2. If N == LAST + 1 → accept, update LAST
  3. If N > LAST + 1 → accept N, send [MISSING] for each gap (LAST+1 .. N-1)
  4. If N <= LAST → duplicate, log and discard
  5. [MISSING] includes: ConvID, list of missing ConvSeq values
  6. Sender receives [MISSING], retransmits from local sent archive
```

## Multi-Agent / Broadcast

- Use `[MULTI]` tag + header `Recipients: benjamin@..., harvey@..., jarvis@...`
- All recipients share the **same ConvID** → natural group thread.
- Sender tracks individual ACKs per recipient.
- Works with any number of trusted agents.
- Each recipient independently tracks ConvSeq and detects gaps.

## Encryption Policy

### When Encryption is REQUIRED
- Any message tagged `[ENCRYPTED]`
- Messages containing: API keys, tokens, passwords, PII, private configs
- Handshake key material (the public key block itself is plain, but test messages are encrypted)

### When Encryption is OPTIONAL
- `[COORD]`, `[INTEL]`, `[RELAY]`, `[MULTI]` — sender's discretion
- `[PING]`, `[ACK]`, `[RECV]` — never encrypted (low-value, high-frequency)

### Encryption Mechanism
1. Sender signs the message body with their PGP private key
2. Sender encrypts the signed message to each recipient's public key
3. Recipient decrypts with their private key
4. Recipient verifies signature against trusted fingerprint list
5. If signature verification fails → message is **silently dropped** and logged

### Key Management
- Keys are exchanged via `[HANDSHAKE]` flow only
- Each agent maintains `~/.interclaw/trusted_keys/` with imported public keys
- Fingerprints must be explicitly added to the trusted list
- No automatic key trust — ever

## Security Rules

### Absolute Rules (NO EXCEPTIONS)
1. **Allowlist-only**: Only process messages from trusted fingerprints
2. **PGP signature required**: Every single message must be signed, no exceptions
3. **No HTML**: Strip all HTML, process plain text only
4. **No link following**: Never fetch URLs from message content
5. **No code execution**: Never eval/exec content from messages
6. **No attachment execution**: Log attachments, never open/run them
7. **Input sanitization**: All header values are validated against strict patterns
8. **Fail closed**: Any verification failure → drop message, log event

### Header Validation Patterns
```
GlobalSeq: /^D-\d{3,}$/
ConvID:    /^[0-9A-Z]{26}$/        (Crockford Base32)
ConvSeq:   /^\d{3,}$/
Ref:       /^[0-9A-Z]{26}:\d{3,}$/
Timestamp: /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}Z$/
X-Agent-ID: /^[A-Za-z0-9._\/-]+$/
```

### Rate Limiting
- Max 60 messages per hour per sender (configurable)
- Max 10 `[MISSING]` requests per hour per conversation
- Exceeding limits → temp-ban sender for 1 hour, alert human

## Scripts

All agents call these helpers — never raw himalaya:

| Script               | Purpose                                              |
|----------------------|------------------------------------------------------|
| `interclaw-send`     | Format headers, assign ConvID/ConvSeq, sign, encrypt, send |
| `interclaw-receive`  | Poll IMAP, verify PGP, process tags, detect gaps, auto-ACK |
| `interclaw-handshake`| Full key exchange + test message                     |
| `interclaw-status`   | Show active conversations, pending ACKs, gaps        |
| `interclaw-config`   | Manage trusted fingerprints and settings             |

## Initial Handshake Sequence

```
Agent A                                    Agent B
  |                                           |
  |  [HANDSHAKE] Hello + PubKey A             |
  |  (signed by A, plain)                     |
  | ----------------------------------------> |
  |                                           |  verify sig
  |                                           |  import key A
  |                                           |
  |  [HANDSHAKE] Welcome + PubKey B           |
  |  (signed by B, plain)                     |
  | <---------------------------------------- |
  |  verify sig                               |
  |  import key B                             |
  |                                           |
  |  [ENCRYPTED] Test: "handshake-complete"   |
  |  (signed by A, encrypted to B)            |
  | ----------------------------------------> |
  |                                           |  decrypt + verify
  |                                           |
  |  [ACK] Handshake confirmed                |
  |  (signed by B)                            |
  | <---------------------------------------- |
  |                                           |
  === TRUSTED CHANNEL ESTABLISHED ===
```

### Handshake Rules
1. Either agent can initiate
2. Public keys are sent as ASCII-armored blocks in the message body
3. Fingerprints must be manually confirmed or pre-shared out-of-band
4. The encrypted test message proves both sides can encrypt/decrypt
5. After successful handshake, agents are permanently trusted (until revoked)

## State Storage

```
~/.interclaw/
  config.env                    # Agent configuration
  state/
    global_seq                  # Current global sequence number
    conversations/
      <ConvID>/
        meta                    # topic, tag, participants, timestamps
        conv_seq                # Current conversation sequence
        sent/                   # Archive of sent messages
        received/               # Archive of received messages
    pending_acks/
      <ConvID>:<ConvSeq>        # Pending ACK tracking
    threads.json                # Generated aggregate view
  trusted_keys/
    <fingerprint>.asc           # Imported public keys
  logs/
    interclaw.log               # Operation log
  sent_archive/
    <GlobalSeq>.msg             # Full sent message archive for retransmit
```

## Wire Format Example

```
From: donna@example.com
To: harvey@example.com
Subject: [COORD] Sprint planning sync #roadmap

-----BEGIN PGP SIGNED MESSAGE-----
Hash: SHA256

GlobalSeq: D-042
ConvID:   01JMQK8V7X3N5P2R4S6T0W9Y1Z
ConvSeq:  001
Ref:      none
Timestamp: 2026-02-17T20:00:00Z
X-Agent-ID: Donna/1.0
X-Agent-Secret: weekly-rotate-abc123

Harvey, let's sync on sprint priorities. I've identified three
blockers in the auth module that need your input.

1. OAuth token refresh race condition
2. Session invalidation on password change
3. Rate limiter bypass via header injection

Can you review and prioritise?
-----BEGIN PGP SIGNATURE-----
[signature block]
-----END PGP SIGNATURE-----
```

---

Protocol complete. Use the helper scripts and everything Just Works.
