---
name: instantdb
description: Real-time database integration with InstantDB. Use this skill when working with InstantDB apps to perform admin operations (create/update/delete entities, link/unlink relationships, query data) and subscribe to real-time data changes. Triggers include mentions of InstantDB, real-time updates, database sync, entity operations, or when OpenClaw needs to send action updates visible to humans in real-time.
---

# InstantDB Integration

## Overview

Node.js integration for InstantDB enabling OpenClaw to perform admin operations and monitor real-time data changes via WebSocket subscriptions.

## Setup

Install dependencies:

```bash
npm install
```

Set environment variables:

```bash
export INSTANTDB_APP_ID="your-app-id"
export INSTANTDB_ADMIN_TOKEN="your-admin-token"
```

## Core Capabilities

### 1. Query Data

Fetch data using InstantDB's query syntax:

```javascript
const { InstantDBClient } = require('./scripts/instantdb.js');

const client = new InstantDBClient(appId, adminToken);
const result = await client.query({
  tasks: {
    $: {
      where: { status: 'active' }
    }
  }
});
```

CLI:
```bash
./scripts/instantdb.js query '{"tasks": {}}'
```

### 2. Create Entities

Add new entities to a namespace:

```javascript
const { entityId, result } = await client.createEntity('tasks', {
  title: 'Process data',
  status: 'pending',
  priority: 'high'
});
```

CLI:
```bash
./scripts/instantdb.js create tasks '{"title": "Process data", "status": "pending"}'
```

Optional entity ID:
```bash
./scripts/instantdb.js create tasks '{"title": "Task"}' custom-entity-id
```

### 3. Update Entities

Modify existing entity attributes:

```javascript
await client.updateEntity(entityId, 'tasks', {
  status: 'completed'
});
```

CLI:
```bash
./scripts/instantdb.js update <entity-id> tasks '{"status": "completed"}'
```

### 4. Delete Entities

Remove entities:

```javascript
await client.deleteEntity(entityId, 'tasks');
```

CLI:
```bash
./scripts/instantdb.js delete <entity-id> tasks
```

### 5. Link Entities

Create relationships between entities:

```javascript
await client.linkEntities(taskId, assigneeId, 'assignees');
```

CLI:
```bash
./scripts/instantdb.js link <parent-id> <child-id> assignees
```

### 6. Unlink Entities

Remove relationships:

```javascript
await client.unlinkEntities(taskId, assigneeId, 'assignees');
```

CLI:
```bash
./scripts/instantdb.js unlink <parent-id> <child-id> assignees
```

### 7. Real-time Subscriptions

Monitor data changes via WebSocket:

```javascript
const subscriptionId = client.subscribe(
  { tasks: { $: { where: { status: 'active' } } } },
  (data) => {
    console.log('Data updated:', data);
  },
  (error) => {
    console.error('Subscription error:', error);
  }
);

// Later: client.unsubscribe(subscriptionId);
```

CLI (listens for specified duration):
```bash
./scripts/instantdb.js subscribe '{"tasks": {}}' 60  # Listen for 60 seconds
```

### 8. Transactions

Execute multiple operations atomically using the tx builder:

```javascript
const { tx, id } = require('@instantdb/admin');

await client.transact([
  tx.tasks[id()].update({ title: 'Task 1' }),
  tx.tasks[id()].update({ title: 'Task 2' })
]);
```

CLI:
```bash
./scripts/instantdb.js transact '[{"op": "update", "id": "...", "data": {...}}]'
```

## OpenClaw Usage Patterns

### Action Status Updates

Send real-time progress to human observers:

```javascript
const { id } = require('@instantdb/admin');

// Create status entity
const actionId = id();
await client.createEntity('actions', {
  type: 'file_processing',
  status: 'started',
  progress: 0,
  timestamp: Date.now()
}, actionId);

// Update progress
await client.updateEntity(actionId, 'actions', {
  progress: 50,
  status: 'processing'
});

// Mark complete
await client.updateEntity(actionId, 'actions', {
  progress: 100,
  status: 'completed'
});
```

### Multi-step Workflow Tracking

Track complex operations:

```javascript
const { tx, id } = require('@instantdb/admin');

const workflowId = id();
const steps = ['Extract', 'Transform', 'Validate', 'Load', 'Verify'];

// Initialize workflow with linked steps
const txs = [
  tx.workflows[workflowId].update({
    name: 'Data Pipeline',
    status: 'running',
    currentStep: 1,
    totalSteps: steps.length
  })
];

const stepIds = steps.map((name, i) => {
  const stepId = id();
  txs.push(
    tx.steps[stepId].update({
      name,
      order: i + 1,
      status: 'pending'
    }),
    tx.workflows[workflowId].link({ steps: stepId })
  );
  return stepId;
});

await client.transact(txs);

// Update as steps complete
for (let i = 0; i < stepIds.length; i++) {
  await client.updateEntity(stepIds[i], 'steps', { 
    status: 'completed' 
  });
  await client.updateEntity(workflowId, 'workflows', { 
    currentStep: i + 2 
  });
}
```

### Human Monitoring Pattern

Humans subscribe to watch OpenClaw's actions:

```javascript
// Human's frontend code
import { init } from '@instantdb/react';

const db = init({ appId });

function ActionMonitor() {
  const { data } = db.useQuery({
    actions: {
      $: {
        where: { status: { in: ['started', 'processing'] } }
      }
    }
  });
  
  return data?.actions?.map(action => (
    <div key={action.id}>
      {action.type}: {action.progress}%
    </div>
  ));
}
```

### Streaming Progress Updates

For long-running operations, stream updates:

```javascript
const { id } = require('@instantdb/admin');

async function processLargeDataset(items) {
  const progressId = id();
  
  await client.createEntity('progress', {
    total: items.length,
    completed: 0,
    status: 'running'
  }, progressId);

  for (let i = 0; i < items.length; i++) {
    // Process item...
    await processItem(items[i]);
    
    // Update every 10 items
    if (i % 10 === 0) {
      await client.updateEntity(progressId, 'progress', {
        completed: i + 1,
        percentage: Math.round(((i + 1) / items.length) * 100)
      });
    }
  }

  await client.updateEntity(progressId, 'progress', {
    completed: items.length,
    percentage: 100,
    status: 'completed'
  });
}
```

## Transaction Patterns

See `references/transactions.md` for detailed transaction patterns including:
- Batch operations
- Relationship management
- Conditional updates
- State machines
- Cascade operations

## Error Handling

All operations return promises that reject on failure:

```javascript
try {
  const result = await client.createEntity('tasks', data);
} catch (error) {
  console.error('Operation failed:', error.message);
}
```

## Query Syntax

See `references/query_syntax.md` for comprehensive query examples including:
- Where clauses and operators
- Relationship traversal
- Sorting and pagination
- Multi-level nesting

## References

- InstantDB documentation: https://www.instantdb.com/docs
- Admin SDK: https://www.instantdb.com/docs/admin
- Query reference: See `references/query_syntax.md`
- Transaction patterns: See `references/transactions.md`

