#!/usr/bin/env python3
"""
Instagram Marketing Product Extractor

Extracts product information from e-commerce URLs for Instagram content generation.
Supports: Amazon, Shopify, Taobao, JD, and generic e-commerce sites.

Usage:
    python extract_product.py <url>

Output:
    JSON object with product details for content generation
"""

import sys
import json
import re
from urllib.parse import urlparse, urlunparse
from typing import Dict, Optional

try:
    import requests
    from bs4 import BeautifulSoup
    REQUESTS_AVAILABLE = True
except ImportError:
    REQUESTS_AVAILABLE = False


class ProductExtractor:
    """Extract product information from e-commerce URLs."""

    def __init__(self, url: str):
        self.url = url
        self.domain = urlparse(url).netloc.lower()
        self.product_data = {}

    def extract(self) -> Dict:
        """Main extraction method - routes to domain-specific extractor."""
        if not REQUESTS_AVAILABLE:
            return self._get_manual_extraction_prompt()

        try:
            headers = {
                'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36'
            }
            response = requests.get(self.url, headers=headers, timeout=15)
            response.raise_for_status()
            soup = BeautifulSoup(response.text, 'html.parser')

            # Route to domain-specific extractor
            if 'amazon' in self.domain:
                return self._extract_amazon(soup)
            elif 'taobao.com' in self.domain or 'tmall.com' in self.domain:
                return self._extract_taobao(soup)
            elif 'jd.com' in self.domain:
                return self._extract_jd(soup)
            elif 'shopify.com' in self.domain:
                return self._extract_shopify(soup)
            else:
                return self._extract_generic(soup)

        except Exception as e:
            return {
                'error': f'Extraction failed: {str(e)}',
                'manual_extraction_needed': True,
                'url': self.url
            }

    def _extract_amazon(self, soup: BeautifulSoup) -> Dict:
        """Extract from Amazon product pages."""
        data = {
            'platform': 'Amazon',
            'url': self.url,
            'name': self._get_text(soup, ['#productTitle', '#title span']),
            'price': self._get_text(soup, ['.a-price-whole']),
            'features': self._get_bullet_points(soup, ['#feature-bullets ul', '#productDescription']),
            'description': self._get_text(soup, ['#productDescription']),
            'images': self._get_images(soup, ['#landingImage', '#altImages img']),
            'rating': self._get_text(soup, ['.a-icon-alt']),
            'review_count': self._get_text(soup, ['#acrCustomerReviewText']),
        }
        return self._clean_data(data)

    def _extract_shopify(self, soup: BeautifulSoup) -> Dict:
        """Extract from Shopify stores."""
        data = {
            'platform': 'Shopify',
            'url': self.url,
            'name': self._get_text(soup, ['h1.product-single__title', '.product_name', 'h1']),
            'price': self._get_text(soup, ['.product-single__price .price', '.current-price']),
            'description': self._get_text(soup, ['.product-single__description', '.product-description']),
            'features': self._get_bullet_points(soup, ['.product-description ul']),
            'images': self._get_images(soup, ['.product-single__photo img', '.product-featured-img']),
        }
        return self._clean_data(data)

    def _extract_taobao(self, soup: BeautifulSoup) -> Dict:
        """Extract from Taobao/Tmall."""
        data = {
            'platform': 'Taobao/Tmall',
            'url': self.url,
            'name': self._get_text(soup, ['.tb-main-title', '.tb-detail-hd h1']),
            'price': self._get_text(soup, ['.tb-price', '.tm-price']),
            'features': self._get_bullet_points(soup, ['.tb-detail-hd p']),
            'images': self._get_images(soup, ['#J_ImgBooth img', '.tb-booth img']),
        }
        return self._clean_data(data)

    def _extract_jd(self, soup: BeautifulSoup) -> Dict:
        """Extract from JD.com."""
        data = {
            'platform': 'JD.com',
            'url': self.url,
            'name': self._get_text(soup, ['.sku-name', '.itemInfo-wrap h1']),
            'price': self._get_text(soup, ['.p-price .price']),
            'features': self._get_bullet_points(soup, ['.detail-content p']),
            'images': self._get_images(soup, ['#spec-img-wrap img', '.imgbox img']),
        }
        return self._clean_data(data)

    def _extract_generic(self, soup: BeautifulSoup) -> Dict:
        """Generic e-commerce extraction."""
        data = {
            'platform': 'Generic',
            'url': self.url,
            'name': self._get_text(soup, ['h1', '.product-title', '.product-name']),
            'price': self._get_text(soup, ['.price', '.product-price', '.product-price']),
            'description': self._get_text(soup, ['.description', '.product-description']),
            'features': self._get_bullet_points(soup, ['ul.features', '.features-list']),
            'images': self._get_images(soup, ['.product-image img', '.product-photo img']),
        }
        return self._clean_data(data)

    def _get_text(self, soup: BeautifulSoup, selectors: list) -> str:
        """Get text from first matching selector."""
        for selector in selectors:
            element = soup.select_one(selector)
            if element:
                text = element.get_text(strip=True)
                return text[:500]  # Limit length
        return ''

    def _get_bullet_points(self, soup: BeautifulSoup, selectors: list) -> list:
        """Extract bullet points from product description."""
        for selector in selectors:
            element = soup.select_one(selector)
            if element:
                bullets = element.find_all('li')
                if bullets:
                    return [b.get_text(strip=True)[:200] for b in bullets[:5]]
        return []

    def _get_images(self, soup: BeautifulSoup, selectors: list) -> list:
        """Extract product image URLs."""
        images = []
        for selector in selectors:
            elements = soup.select(selector)
            for element in elements[:5]:
                src = element.get('src') or element.get('data-src')
                if src:
                    if src.startswith('//'):
                        src = 'https:' + src
                    elif src.startswith('/'):
                        src = f"https://{self.domain}{src}"
                    images.append(src)
        return images

    def _clean_data(self, data: Dict) -> Dict:
        """Clean and standardize extracted data."""
        cleaned = {
            'platform': data.get('platform', 'Unknown'),
            'url': data.get('url', self.url),
            'name': data.get('name', '').strip(),
            'price': data.get('price', ''),
            'description': data.get('description', '')[:1000],
            'features': data.get('features', [])[:5],
            'images': data.get('images', [])[:5],
        }

        # Add analysis fields
        cleaned['target_audience'] = self._infer_audience(cleaned)
        cleaned['usp'] = self._infer_usp(cleaned)
        cleaned['content_type'] = self._suggest_content_type(cleaned)

        return cleaned

    def _infer_audience(self, data: Dict) -> str:
        """Infer target audience from product name and description."""
        text = f"{data.get('name', '')} {data.get('description', '')}".lower()

        if any(word in text for word in ['men', 'male', 'guy', 'gentleman']):
            return 'Men'
        elif any(word in text for word in ['women', 'female', 'lady', 'girl']):
            return 'Women'
        elif any(word in text for word in ['kids', 'children', 'baby', 'toddler']):
            return 'Kids/Family'
        elif any(word in text for word in ['gaming', 'gamer', 'esports']):
            return 'Gamers'
        elif any(word in text for word in ['fitness', 'workout', 'exercise', 'sport']):
            return 'Fitness Enthusiasts'
        elif any(word in text for word in ['home', 'decor', 'kitchen', 'furniture']):
            return 'Homeowners'
        else:
            return 'General Consumers'

    def _infer_usp(self, data: Dict) -> str:
        """Infer unique selling proposition."""
        name = data.get('name', '').lower()
        features = ' '.join(data.get('features', []))

        if 'organic' in name or 'natural' in name:
            return 'Made with organic/natural ingredients'
        elif 'handmade' in name or 'artisan' in name:
            return 'Handcrafted with care'
        elif any(word in features for word in ['premium', 'luxury', 'high-quality']):
            return 'Premium quality product'
        elif 'wireless' in name or 'bluetooth' in name:
            return 'Convenient wireless design'
        else:
            return 'Great value for money'

    def _suggest_content_type(self, data: Dict) -> str:
        """Suggest best Instagram content format."""
        name = data.get('name', '').lower()
        features = data.get('features', [])

        if 'fashion' in name or 'clothing' in name or 'shoe' in name:
            return 'Carousel (Lifestyle + Details + Fit)'
        elif any(word in name for word in ['video', 'camera', 'audio', 'speaker']):
            return 'Reel (Demo + Usage Scenes)'
        elif len(features) >= 3:
            return 'Carousel (Feature-by-feature breakdown)'
        elif data.get('images'):
            return 'Feed Post (High-quality product shot)'
        else:
            return 'Feed Post (Lifestyle with product)'

    def _get_manual_extraction_prompt(self) -> Dict:
        """Return prompt for manual extraction when libraries unavailable."""
        return {
            'error': 'Required libraries not available. Please install: pip install requests beautifulsoup4',
            'manual_extraction_needed': True,
            'url': self.url,
            'extraction_guide': {
                'product_name': 'Copy from page title or h1',
                'price': 'Find price display on page',
                'features': 'Look for bullet points or product highlights (3-5 items)',
                'description': 'Copy product description (first paragraph)',
                'target_audience': 'Who would buy this? (Men/Women/Kids/etc)',
                'brand_tone': 'Is it luxury, playful, minimal, or bold?',
                'existing_images': 'Note if there are product photos to reference'
            }
        }


def main():
    if len(sys.argv) < 2:
        print(json.dumps({
            'error': 'Usage: python extract_product.py <url>',
            'example': 'python extract_product.py https://www.amazon.com/dp/B08N5WRWNW'
        }, indent=2))
        sys.exit(1)

    url = sys.argv[1]
    extractor = ProductExtractor(url)
    result = extractor.extract()

    print(json.dumps(result, indent=2, ensure_ascii=False))


if __name__ == '__main__':
    main()
