# 视觉一致性 (Visual Consistency) — AI 生成的核心挑战

> **核心原则**: AI 生成与物理拍摄的根本区别在于，AI 每次生成都是独立任务，默认无法保证物体、场景、环境的视觉一致性。

---

## 问题本质

### AI 生成 vs. 物理拍摄

| 特性 | 物理拍摄 | AI 生成 |
|------|---------|---------|
| **一致性来源** | 同一物理场景/演员 | 每次独立采样 |
| **默认行为** | 自动一致 | 默认不一致 |
| **控制方式** | 场景搭建、演员定妆 | 参考图、种子控制 |
| **典型问题** | 光线变化、演员疲劳 | 物体外观变化、场景突变 |

**示例问题**:
- 同一个产品在不同图片中颜色/形状不一致
- 同一角色在视频不同镜头中外观变化
- 背景场景在系列内容中风格差异大

---

## 识别视觉一致性需求

### 用户隐含需求识别

**关键词触发**:
- "系列图片/视频" → 需要一致性
- "同一个产品/角色/场景" → 需要一致性
- "多张图片" / "分镜头" → 需要一致性
- "续集" / "延续" / "接上" → 需要一致性

**隐含需求识别**:
```
用户: "帮我生成3张产品海报"
  ↓ 分析
产品应该是同一个 → 需要视觉一致性
  ↓ 正确方案
第1张: 文生图 → 保存参考图
第2-3张: 图生图 (使用第1张作为参考)
```

```
用户: "生成一个角色，然后让他做不同动作"
  ↓ 分析
角色外观必须一致 → 需要视觉一致性
  ↓ 正确方案
1. 生成角色设定图 (正面、侧面、背面)
2. 所有后续生成使用角色参考图
```

### 一致性需求分类

**Level 1: 物体/角色一致性**
- 同一产品在不同场景中
- 同一角色在不同动作/表情中
- 同一 IP 形象的延续

**Level 2: 场景/环境一致性**
- 同一室内/户外场景的不同角度
- 同一时间/天气/光线条件
- 同一艺术风格/色调

**Level 3: 故事连贯性**
- 视频分镜的连续性
- 系列内容的风格统一
- 品牌视觉规范

---

## 技术解决方案

### ⭐ 核心策略: 参考图驱动

**正确流程**:
```
1. 首次生成 (文生图/视频)
   ↓
   生成结果 A (保存作为参考图)
   
2. 后续生成 (图生图/视频)
   ↓
   参考图 A + 新提示词
   ↓
   生成结果 B (保持 A 的视觉元素)
   
3. 持续迭代
   ↓
   参考图 A + 参考图 B + 新提示词
   ↓
   生成结果 C (保持系列一致性)
```

### 错误做法 ❌

**反例 1: 纯文本多次生成**
```python
# ❌ 错误: 每次独立生成
for i in range(3):
    generate_image(prompt="红色运动鞋，白色背景")
    # 结果: 3 双不同的鞋子
```

**反例 2: 忽视一致性需求**
```python
# ❌ 错误: 未识别用户的隐含需求
用户: "生成产品的3个使用场景"
→ 直接生成3张独立图片
→ 产品外观完全不同
```

### 正确做法 ✅

**方案 1: 参考图锁定主体**
```python
# ✅ 正确: 使用参考图保持一致性
# Step 1: 生成基础图
base_image = generate_image(
    prompt="红色运动鞋产品照，白色背景，专业摄影"
)

# Step 2: 基于参考图生成变体
scene1 = generate_image(
    prompt="运动鞋放在健身房场景",
    reference_image=base_image,
    reference_strength=0.8  # 高强度保持外观
)

scene2 = generate_image(
    prompt="运动鞋户外跑步场景",
    reference_image=base_image,
    reference_strength=0.8
)
```

**方案 2: 角色设定图**
```python
# ✅ 正确: 先生成角色设定，再生成场景
# Step 1: 生成角色设定图
character_sheet = generate_image(
    prompt="卡通女孩，正面、侧面、背面三视图，角色设计"
)

# Step 2: 基于角色设定生成动作
action1 = generate_image(
    prompt="卡通女孩在跳舞",
    reference_image=character_sheet,
    reference_mode="character"
)

action2 = generate_image(
    prompt="卡通女孩在读书",
    reference_image=character_sheet,
    reference_mode="character"
)
```

**方案 3: 视频分镜一致性**
```python
# ✅ 正确: 使用首帧作为参考
# Step 1: 生成第一镜头
shot1 = generate_video(
    prompt="咖啡店内景，温暖光线，顾客入座"
)

# Step 2: 提取关键帧作为参考
reference_frame = extract_frame(shot1, timestamp=3.0)

# Step 3: 生成后续镜头
shot2 = generate_video(
    prompt="咖啡师制作咖啡特写",
    reference_image=reference_frame,
    reference_strength=0.6  # 保持场景氛围
)
```

---

## IMA Studio 参数支持

### 图像生成参数

**参考图控制** (Image-to-Image):
```json
{
  "reference_image": "url_or_base64",
  "reference_strength": 0.5-1.0,  // 0.8+ 强一致性
  "reference_mode": "character|style|composition"
}
```

**种子控制** (部分模型支持):
```json
{
  "seed": 123456,  // 固定种子获得相似结果
  "variation_seed": 789012  // 微调变化
}
```

### 视频生成参数

**首帧/尾帧控制**:
```json
{
  "first_frame": "reference_image_url",  // 控制开始画面
  "last_frame": "reference_image_url",   // 控制结束画面
  "frame_reference_strength": 0.7        // 帧参考强度
}
```

**镜头连贯性**:
```json
{
  "previous_clip": "video_url",          // 接续上一镜头
  "transition": "cut|fade|dissolve",     // 转场方式
  "match_color_grade": true              // 匹配色调
}
```

---

## 工作流模板

### 模板 1: 产品系列图生成

```
用户需求: "生成产品的3个使用场景"

工作流:
1. [ima-image-ai] 生成产品主图
   - 模型: Nano Banana Pro 4K
   - 提示词: "产品专业摄影，白色背景，高清"
   - 保存: product_master.jpg

2. [ima-image-ai] 生成场景1
   - 模型: Nano Banana Pro 4K
   - 提示词: "产品在客厅使用场景"
   - 参考图: product_master.jpg
   - 参考强度: 0.85

3. [ima-image-ai] 生成场景2
   - 模型: Nano Banana Pro 4K
   - 提示词: "产品在办公室使用场景"
   - 参考图: product_master.jpg
   - 参考强度: 0.85

4. [ima-image-ai] 生成场景3
   - 模型: Nano Banana Pro 4K
   - 提示词: "产品户外使用场景"
   - 参考图: product_master.jpg
   - 参考强度: 0.85

输出: 3张视觉一致的产品场景图
```

### 模板 2: 角色系列动作

```
用户需求: "生成一个角色做不同动作"

工作流:
1. [ima-image-ai] 生成角色设定图
   - 模型: SeeDream 4.5
   - 提示词: "卡通角色设定，正面+侧面+背面，三视图"
   - 保存: character_sheet.jpg

2. [ima-image-ai] 生成动作1 (站立)
   - 参考图: character_sheet.jpg
   - 参考模式: character
   - 提示词: "角色站立姿势，全身"

3. [ima-image-ai] 生成动作2 (跑步)
   - 参考图: character_sheet.jpg
   - 参考模式: character
   - 提示词: "角色跑步动作，全身"

4. [ima-image-ai] 生成动作3 (跳跃)
   - 参考图: character_sheet.jpg
   - 参考模式: character
   - 提示词: "角色跳跃动作，全身"

输出: 同一角色的不同动作图
```

### 模板 3: 视频分镜连贯性

```
用户需求: "生成一个品牌宣传视频，3个镜头"

工作流:
1. [设计阶段] 规划视觉一致性
   - 确定: 色调、光线、场景风格
   - 生成: 视觉风格参考图

2. [ima-video-ai] 生成镜头1 (开场)
   - 模型: CogVideoX-5B
   - 提示词: "品牌logo展示，现代简约风格"
   - 保存关键帧: scene_style.jpg

3. [ima-video-ai] 生成镜头2 (产品展示)
   - 模型: CogVideoX-5B
   - 提示词: "产品特写，旋转展示"
   - 参考图: scene_style.jpg (保持色调)
   - 首帧: 镜头1的尾帧 (无缝衔接)

4. [ima-video-ai] 生成镜头3 (结尾)
   - 模型: CogVideoX-5B
   - 提示词: "品牌slogan文字动画"
   - 参考图: scene_style.jpg
   - 首帧: 镜头2的尾帧

输出: 3个视觉连贯的视频片段
```

---

## 常见错误与修复

### 错误 1: 未识别一致性需求

**症状**:
```
用户: "生成5张产品海报"
→ 生成了5个完全不同的产品
→ 用户: "怎么产品都不一样？"
```

**诊断**:
- 未识别"同一产品"的隐含需求
- 使用纯文生图，每次独立生成

**修复**:
1. 先确认: "这5张海报是同一个产品吗？"
2. 如果是，使用参考图工作流
3. 第1张文生图，2-5张图生图

---

### 错误 2: 参考强度设置不当

**症状**:
```
reference_strength = 0.3
→ 参考图影响太弱，一致性不足

reference_strength = 1.0
→ 参考图影响过强，无法生成新场景
```

**诊断**:
- 参考强度是平衡点: 一致性 vs. 创造性
- 过低: 失去一致性
- 过高: 失去灵活性

**修复**:
- **高一致性需求**: 0.8-0.9 (产品外观、角色面容)
- **中等一致性**: 0.6-0.7 (场景风格、色调氛围)
- **低一致性**: 0.4-0.5 (灵感参考、风格借鉴)

---

### 错误 3: 忽视视频分镜连贯性

**症状**:
```
镜头1: 白天咖啡店
镜头2: 突然变成夜晚酒吧
→ 用户: "为什么场景跳跃这么大？"
```

**诊断**:
- 每个镜头独立生成，未考虑连贯性
- 未使用首帧/尾帧控制

**修复**:
1. 规划视频时，先设计视觉连贯性
2. 使用 `first_frame` 参数连接镜头
3. 统一色调、光线、场景风格
4. 添加转场效果 (fade/dissolve)

---

## 最佳实践总结

### ✅ DO (推荐做法)

1. **提前识别一致性需求**
   - 主动询问: "这些内容是同一主体吗？"
   - 识别关键词: "系列"、"同一"、"延续"

2. **优先使用参考图**
   - 首次生成: 文生图/视频
   - 后续生成: 图生图/视频 + 参考图

3. **保存中间结果**
   - 保存基础图作为参考
   - 保存关键帧用于后续生成

4. **参数化一致性控制**
   - 使用 `reference_strength` 微调一致性
   - 使用 `seed` 固定风格 (部分模型)

5. **明确告知用户**
   - 说明为什么使用参考图
   - 解释一致性控制的重要性

### ❌ DON'T (避免做法)

1. ❌ 纯文本多次生成 → 每次结果不同
2. ❌ 忽视用户隐含需求 → 生成不符合预期
3. ❌ 参考强度设置极端 (0.1 或 1.0)
4. ❌ 视频分镜独立生成 → 镜头不连贯
5. ❌ 不保存中间结果 → 无法追溯一致性

---

## 总结

> **核心原则**: AI 生成默认不具备视觉一致性，必须通过参考图主动控制。

**关键要点**:
1. ✅ **识别一致性需求** — 主动询问，识别隐含需求
2. ✅ **参考图驱动** — 首次文生图，后续图生图
3. ✅ **参数化控制** — 调整 reference_strength 平衡一致性与创造性
4. ✅ **工作流设计** — 先生成基础图，再生成变体
5. ✅ **用户沟通** — 解释为什么使用参考图，管理预期

**记住**:
- **物理拍摄** = 天然一致性 (同一场景/演员)
- **AI 生成** = 默认不一致 (独立采样)
- **解决方案** = 参考图控制 (Image-to-Image / Video-to-Video)

---

**实战建议**:
当用户说"生成多张图片/多个镜头"时，**默认假设需要视觉一致性**，除非明确说明需要完全不同的内容。

**快速检查清单**:
- [ ] 用户是否需要"同一主体"？
- [ ] 是否是系列内容/分镜头？
- [ ] 是否需要保持场景/风格一致？
- [ ] 如果是，是否使用了参考图？
- [ ] 参考强度是否合理 (0.6-0.9)？

---

**最后更新**: 2026-03-02  
**来源**: IMA Studio Production Best Practices (硕哥指导)
