#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
i18n - Internationalization support
Supports: English, Spanish, French, German, Italian, Russian, Dutch, Swedish, Finnish, Danish, Ukrainian

@author: Black_Temple
@version: 2.3.0
"""

import os
import locale
from typing import Dict

# Supported languages
SUPPORTED_LANGUAGES = ['en', 'es', 'fr', 'de', 'it', 'ru', 'nl', 'sv', 'fi', 'da', 'uk']

# Translation dictionaries
TRANSLATIONS: Dict[str, Dict[str, str]] = {
    'en': {
        # Setup
        'setup_title': '🔧 iCalendar Sync Setup',
        'setup_intro': 'To use iCalendar Sync, you need to configure your iCloud credentials.',
        'setup_warning': '⚠️  Use an App-Specific Password, NOT your regular Apple ID password.',
        'setup_url': 'Get it from: https://appleid.apple.com -> Sign-In & Security -> App-Specific Passwords',
        'email_prompt': '📧 iCloud Email: ',
        'password_prompt': '🔑 App-Specific Password (xxxx-xxxx-xxxx-xxxx): ',
        'email_empty': '❌ Email cannot be empty',
        'password_empty': '❌ Password cannot be empty',
        'invalid_email': '❌ Invalid email format',
        'continue_anyway': 'Continue anyway? (y/n): ',
        'password_unusual': '⚠️  Password format looks unusual',
        'are_you_sure': 'Are you sure this is correct? (y/n): ',
        'setup_cancelled': 'Setup cancelled',
        'credentials_saved_keyring': '✅ Credentials saved securely to system keyring',
        'keyring_fallback': '⚠️  Could not access system keyring, falling back to .env file',
        'credentials_saved_file': '✅ Configuration saved securely to {path}',
        'ready_to_use': '🚀 You can now use iCalendar Sync!',
        'available_calendars': '📅 Available Calendars ({count}):',
        'events_in_calendar': '📋 Events in \'{name}\' ({count} found):',
        'event_created': '✅ Event \'{summary}\' created successfully',
        'event_deleted': '🗑️  Event deleted successfully',
        'error_required_packages': '❌ Required packages not installed: {error}',
        'error_run_install': 'Run: pip install -r requirements.txt',
        'error_credentials_missing': '❌ iCloud credentials not configured',
        'error_auth_failed': '❌ Authentication failed: Invalid credentials',
        'error_network': '❌ Network error',
        'error_caldav': '❌ CalDAV error',
        'error_unexpected': '❌ Unexpected error: {type}',
        'error_calendars_not_found': '❌ Calendars not found',
        'error_calendar_not_found': '❌ Calendar \'{name}\' not found',
        'error_events': '❌ Error getting events',
        'error_listing_calendars': '❌ Error listing calendars',
        'error_invalid_calendar_name': '❌ Invalid calendar name',
        'error_days_range': '❌ days_ahead must be between {min} and {max}',
        'error_missing_fields': '❌ Missing required fields: {fields}',
        'error_dtstart_type': '❌ dtstart must be a datetime object',
        'error_dtend_type': '❌ dtend must be a datetime object',
        'error_time_range': '❌ Event end time must be after start time',
        'error_creating_event': '❌ Error creating event',
        'error_event_not_found': '❌ Event or calendar not found',
        'error_deleting_event': '❌ Error deleting event',
        'error_invalid_uid': '❌ Valid event UID required',
        'error_uid_too_long': '❌ Invalid event UID (too long)',
        'error_calendar_required': '❌ Calendar name required',
        'error_json_required': '❌ Calendar and JSON data required',
        'error_json_read': '❌ Could not read JSON file',
        'error_json_too_large': '❌ JSON data too large',
        'error_invalid_json': '❌ Invalid JSON: {error}',
        'error_invalid_datetime': '❌ Invalid datetime format: {error}',
        'error_uid_required': '❌ Calendar and event UID required',
        'warning_conflicts': '⚠️  Warning: {count} conflicting event(s) found:',
        'warning_conflict_item': '   - {summary} ({start} to {end})',
        'confirm_continue': 'Continue anyway? (y/n): ',
        'creation_cancelled': 'Event creation cancelled',
        'input_timeout': '⏱️  Input timeout',
        'using_cached_connection': 'Using cached CalDAV connection',
        'connection_successful': 'Successfully connected to iCloud CalDAV',
        'found_calendar': 'Found calendar: {name}',
        'found_event': 'Found event: {summary}',
        'loaded_from_keyring': 'Loaded password from keyring',
    },
    
    'ru': {
        'setup_title': '🔧 Настройка iCalendar Sync',
        'setup_intro': 'Для использования iCalendar Sync необходимо настроить учетные данные iCloud.',
        'setup_warning': '⚠️  Используйте пароль для приложения, НЕ обычный пароль Apple ID.',
        'setup_url': 'Получить здесь: https://appleid.apple.com -> Вход и безопасность -> Пароли приложений',
        'email_prompt': '📧 Email iCloud: ',
        'password_prompt': '🔑 Пароль для приложения (xxxx-xxxx-xxxx-xxxx): ',
        'email_empty': '❌ Email не может быть пустым',
        'password_empty': '❌ Пароль не может быть пустым',
        'invalid_email': '❌ Неверный формат email',
        'continue_anyway': 'Продолжить всё равно? (y/n): ',
        'password_unusual': '⚠️  Формат пароля выглядит необычно',
        'are_you_sure': 'Вы уверены, что это правильно? (y/n): ',
        'setup_cancelled': 'Настройка отменена',
        'credentials_saved_keyring': '✅ Учетные данные безопасно сохранены в системной связке ключей',
        'keyring_fallback': '⚠️  Не удалось получить доступ к связке ключей, используется файл .env',
        'credentials_saved_file': '✅ Конфигурация безопасно сохранена в {path}',
        'ready_to_use': '🚀 Теперь вы можете использовать iCalendar Sync!',
        'available_calendars': '📅 Доступные календари ({count}):',
        'events_in_calendar': '📋 События в \'{name}\' (найдено {count}):',
        'event_created': '✅ Событие \'{summary}\' успешно создано',
        'event_deleted': '🗑️  Событие успешно удалено',
        'error_required_packages': '❌ Необходимые пакеты не установлены: {error}',
        'error_run_install': 'Выполните: pip install -r requirements.txt',
        'error_credentials_missing': '❌ Учетные данные iCloud не настроены',
        'error_auth_failed': '❌ Ошибка аутентификации: неверные учетные данные',
        'error_network': '❌ Ошибка сети',
        'error_caldav': '❌ Ошибка CalDAV',
        'error_unexpected': '❌ Неожиданная ошибка: {type}',
        'error_calendars_not_found': '❌ Календари не найдены',
        'error_calendar_not_found': '❌ Календарь \'{name}\' не найден',
        'error_events': '❌ Ошибка получения событий',
        'error_listing_calendars': '❌ Ошибка получения списка календарей',
        'error_invalid_calendar_name': '❌ Неверное имя календаря',
        'error_days_range': '❌ days_ahead должно быть от {min} до {max}',
        'error_missing_fields': '❌ Отсутствуют обязательные поля: {fields}',
        'error_dtstart_type': '❌ dtstart должен быть объектом datetime',
        'error_dtend_type': '❌ dtend должен быть объектом datetime',
        'error_time_range': '❌ Время окончания события должно быть после времени начала',
        'error_creating_event': '❌ Ошибка создания события',
        'error_event_not_found': '❌ Событие или календарь не найдены',
        'error_deleting_event': '❌ Ошибка удаления события',
        'error_invalid_uid': '❌ Требуется корректный UID события',
        'error_uid_too_long': '❌ Неверный UID события (слишком длинный)',
        'error_calendar_required': '❌ Требуется имя календаря',
        'error_json_required': '❌ Требуются календарь и данные JSON',
        'error_json_read': '❌ Не удалось прочитать файл JSON',
        'error_json_too_large': '❌ Данные JSON слишком большие',
        'error_invalid_json': '❌ Неверный JSON: {error}',
        'error_invalid_datetime': '❌ Неверный формат даты/времени: {error}',
        'error_uid_required': '❌ Требуются календарь и UID события',
        'warning_conflicts': '⚠️  Предупреждение: найдено {count} конфликтующих событий:',
        'warning_conflict_item': '   - {summary} ({start} до {end})',
        'confirm_continue': 'Продолжить всё равно? (y/n): ',
        'creation_cancelled': 'Создание события отменено',
        'input_timeout': '⏱️  Время ожидания ввода истекло',
        'using_cached_connection': 'Используется кэшированное подключение CalDAV',
        'connection_successful': 'Успешно подключено к iCloud CalDAV',
        'found_calendar': 'Найден календарь: {name}',
        'found_event': 'Найдено событие: {summary}',
        'loaded_from_keyring': 'Пароль загружен из связки ключей',
    },
    
    'nl': {
        'setup_title': '🔧 iCalendar Sync Installatie',
        'setup_intro': 'Om iCalendar Sync te gebruiken, moet u uw iCloud-inloggegevens configureren.',
        'setup_warning': '⚠️  Gebruik een App-specifiek wachtwoord, NIET uw gewone Apple ID-wachtwoord.',
        'setup_url': 'Verkrijg het van: https://appleid.apple.com -> Aanmelden en beveiliging -> App-specifieke wachtwoorden',
        'email_prompt': '📧 iCloud E-mail: ',
        'password_prompt': '🔑 App-specifiek wachtwoord (xxxx-xxxx-xxxx-xxxx): ',
        'email_empty': '❌ E-mail mag niet leeg zijn',
        'password_empty': '❌ Wachtwoord mag niet leeg zijn',
        'invalid_email': '❌ Ongeldig e-mailformaat',
        'continue_anyway': 'Toch doorgaan? (j/n): ',
        'password_unusual': '⚠️  Wachtwoordformaat ziet er ongebruikelijk uit',
        'are_you_sure': 'Weet u zeker dat dit correct is? (j/n): ',
        'setup_cancelled': 'Installatie geannuleerd',
        'credentials_saved_keyring': '✅ Inloggegevens veilig opgeslagen in systeemsleutelhanger',
        'keyring_fallback': '⚠️  Kon geen toegang krijgen tot systeemsleutelhanger, gebruik .env-bestand',
        'credentials_saved_file': '✅ Configuratie veilig opgeslagen in {path}',
        'ready_to_use': '🚀 U kunt nu iCalendar Sync gebruiken!',
        'available_calendars': '📅 Beschikbare Agenda\'s ({count}):',
        'events_in_calendar': '📋 Gebeurtenissen in \'{name}\' ({count} gevonden):',
        'event_created': '✅ Gebeurtenis \'{summary}\' succesvol aangemaakt',
        'event_deleted': '🗑️  Gebeurtenis succesvol verwijderd',
        'error_required_packages': '❌ Vereiste pakketten niet geïnstalleerd: {error}',
        'error_run_install': 'Voer uit: pip install -r requirements.txt',
        'error_credentials_missing': '❌ iCloud-inloggegevens niet geconfigureerd',
        'error_auth_failed': '❌ Authenticatie mislukt: ongeldige inloggegevens',
        'error_network': '❌ Netwerkfout',
        'error_caldav': '❌ CalDAV-fout',
        'error_unexpected': '❌ Onverwachte fout: {type}',
        'error_calendars_not_found': '❌ Agenda\'s niet gevonden',
        'error_calendar_not_found': '❌ Agenda \'{name}\' niet gevonden',
        'error_events': '❌ Fout bij ophalen gebeurtenissen',
        'error_listing_calendars': '❌ Fout bij weergeven agenda\'s',
        'error_invalid_calendar_name': '❌ Ongeldige agendanaam',
        'error_days_range': '❌ days_ahead moet tussen {min} en {max} zijn',
        'error_missing_fields': '❌ Ontbrekende verplichte velden: {fields}',
        'error_dtstart_type': '❌ dtstart moet een datetime-object zijn',
        'error_dtend_type': '❌ dtend moet een datetime-object zijn',
        'error_time_range': '❌ Eindtijd gebeurtenis moet na starttijd zijn',
        'error_creating_event': '❌ Fout bij aanmaken gebeurtenis',
        'error_event_not_found': '❌ Gebeurtenis of agenda niet gevonden',
        'error_deleting_event': '❌ Fout bij verwijderen gebeurtenis',
        'error_invalid_uid': '❌ Geldige gebeurtenis-UID vereist',
        'error_uid_too_long': '❌ Ongeldige gebeurtenis-UID (te lang)',
        'error_calendar_required': '❌ Agendanaam vereist',
        'error_json_required': '❌ Agenda en JSON-gegevens vereist',
        'error_json_read': '❌ Kon JSON-bestand niet lezen',
        'error_json_too_large': '❌ JSON-gegevens te groot',
        'error_invalid_json': '❌ Ongeldige JSON: {error}',
        'error_invalid_datetime': '❌ Ongeldig datum/tijd-formaat: {error}',
        'error_uid_required': '❌ Agenda en gebeurtenis-UID vereist',
        'warning_conflicts': '⚠️  Waarschuwing: {count} conflicterende gebeurtenis(sen) gevonden:',
        'warning_conflict_item': '   - {summary} ({start} tot {end})',
        'confirm_continue': 'Toch doorgaan? (j/n): ',
        'creation_cancelled': 'Aanmaken gebeurtenis geannuleerd',
        'input_timeout': '⏱️  Invoer time-out',
        'using_cached_connection': 'Gebruik van gecachte CalDAV-verbinding',
        'connection_successful': 'Succesvol verbonden met iCloud CalDAV',
        'found_calendar': 'Agenda gevonden: {name}',
        'found_event': 'Gebeurtenis gevonden: {summary}',
        'loaded_from_keyring': 'Wachtwoord geladen uit sleutelhanger',
    },
    
    'sv': {
        'setup_title': '🔧 iCalendar Sync Installation',
        'setup_intro': 'För att använda iCalendar Sync måste du konfigurera dina iCloud-autentiseringsuppgifter.',
        'setup_warning': '⚠️  Använd ett App-specifikt lösenord, INTE ditt vanliga Apple ID-lösenord.',
        'setup_url': 'Hämta det från: https://appleid.apple.com -> Inloggning och säkerhet -> App-specifika lösenord',
        'email_prompt': '📧 iCloud E-post: ',
        'password_prompt': '🔑 App-specifikt lösenord (xxxx-xxxx-xxxx-xxxx): ',
        'email_empty': '❌ E-post kan inte vara tom',
        'password_empty': '❌ Lösenord kan inte vara tomt',
        'invalid_email': '❌ Ogiltigt e-postformat',
        'continue_anyway': 'Fortsätt ändå? (j/n): ',
        'password_unusual': '⚠️  Lösenordsformatet ser ovanligt ut',
        'are_you_sure': 'Är du säker på att detta är korrekt? (j/n): ',
        'setup_cancelled': 'Installation avbruten',
        'credentials_saved_keyring': '✅ Autentiseringsuppgifter säkert sparade i systemnyckelring',
        'keyring_fallback': '⚠️  Kunde inte komma åt systemnyckelring, använder .env-fil',
        'credentials_saved_file': '✅ Konfiguration säkert sparad i {path}',
        'ready_to_use': '🚀 Du kan nu använda iCalendar Sync!',
        'available_calendars': '📅 Tillgängliga Kalendrar ({count}):',
        'events_in_calendar': '📋 Händelser i \'{name}\' ({count} hittade):',
        'event_created': '✅ Händelse \'{summary}\' skapad framgångsrikt',
        'event_deleted': '🗑️  Händelse raderad framgångsrikt',
        'error_required_packages': '❌ Nödvändiga paket inte installerade: {error}',
        'error_run_install': 'Kör: pip install -r requirements.txt',
        'error_credentials_missing': '❌ iCloud-autentiseringsuppgifter inte konfigurerade',
        'error_auth_failed': '❌ Autentisering misslyckades: ogiltiga autentiseringsuppgifter',
        'error_network': '❌ Nätverksfel',
        'error_caldav': '❌ CalDAV-fel',
        'error_unexpected': '❌ Oväntat fel: {type}',
        'error_calendars_not_found': '❌ Kalendrar hittades inte',
        'error_calendar_not_found': '❌ Kalender \'{name}\' hittades inte',
        'error_events': '❌ Fel vid hämtning av händelser',
        'error_listing_calendars': '❌ Fel vid listning av kalendrar',
        'error_invalid_calendar_name': '❌ Ogiltigt kalendernamn',
        'error_days_range': '❌ days_ahead måste vara mellan {min} och {max}',
        'error_missing_fields': '❌ Saknade obligatoriska fält: {fields}',
        'error_dtstart_type': '❌ dtstart måste vara ett datetime-objekt',
        'error_dtend_type': '❌ dtend måste vara ett datetime-objekt',
        'error_time_range': '❌ Händelsens sluttid måste vara efter starttid',
        'error_creating_event': '❌ Fel vid skapande av händelse',
        'error_event_not_found': '❌ Händelse eller kalender hittades inte',
        'error_deleting_event': '❌ Fel vid radering av händelse',
        'error_invalid_uid': '❌ Giltig händelse-UID krävs',
        'error_uid_too_long': '❌ Ogiltig händelse-UID (för lång)',
        'error_calendar_required': '❌ Kalendernamn krävs',
        'error_json_required': '❌ Kalender och JSON-data krävs',
        'error_json_read': '❌ Kunde inte läsa JSON-fil',
        'error_json_too_large': '❌ JSON-data för stor',
        'error_invalid_json': '❌ Ogiltig JSON: {error}',
        'error_invalid_datetime': '❌ Ogiltigt datum/tid-format: {error}',
        'error_uid_required': '❌ Kalender och händelse-UID krävs',
        'warning_conflicts': '⚠️  Varning: {count} konflikterande händelse(r) hittades:',
        'warning_conflict_item': '   - {summary} ({start} till {end})',
        'confirm_continue': 'Fortsätt ändå? (j/n): ',
        'creation_cancelled': 'Skapande av händelse avbrutet',
        'input_timeout': '⏱️  Inmatnings-timeout',
        'using_cached_connection': 'Använder cachad CalDAV-anslutning',
        'connection_successful': 'Framgångsrikt ansluten till iCloud CalDAV',
        'found_calendar': 'Hittade kalender: {name}',
        'found_event': 'Hittade händelse: {summary}',
        'loaded_from_keyring': 'Lösenord laddat från nyckelring',
    },
    
    'fi': {
        'setup_title': '🔧 iCalendar Sync Asennus',
        'setup_intro': 'Käyttääksesi iCalendar Synciä, sinun on määritettävä iCloud-tunnuksesi.',
        'setup_warning': '⚠️  Käytä sovellukselle tarkoitettua salasanaa, ÄLKÄÄ tavallista Apple ID -salasanaa.',
        'setup_url': 'Hanki se: https://appleid.apple.com -> Kirjautuminen ja turvallisuus -> Sovelluksille tarkoitetut salasanat',
        'email_prompt': '📧 iCloud Sähköposti: ',
        'password_prompt': '🔑 Sovellukselle tarkoitettu salasana (xxxx-xxxx-xxxx-xxxx): ',
        'email_empty': '❌ Sähköposti ei voi olla tyhjä',
        'password_empty': '❌ Salasana ei voi olla tyhjä',
        'invalid_email': '❌ Virheellinen sähköpostimuoto',
        'continue_anyway': 'Jatka silti? (k/e): ',
        'password_unusual': '⚠️  Salasanan muoto näyttää epätavalliselta',
        'are_you_sure': 'Oletko varma, että tämä on oikein? (k/e): ',
        'setup_cancelled': 'Asennus peruutettu',
        'credentials_saved_keyring': '✅ Tunnukset tallennettu turvallisesti järjestelmän avainnippuun',
        'keyring_fallback': '⚠️  Järjestelmän avainnippuun pääsy epäonnistui, käytetään .env-tiedostoa',
        'credentials_saved_file': '✅ Määritys tallennettu turvallisesti kohteeseen {path}',
        'ready_to_use': '🚀 Voit nyt käyttää iCalendar Synciä!',
        'available_calendars': '📅 Saatavilla olevat Kalenterit ({count}):',
        'events_in_calendar': '📋 Tapahtumat kalenterissa \'{name}\' ({count} löydetty):',
        'event_created': '✅ Tapahtuma \'{summary}\' luotu onnistuneesti',
        'event_deleted': '🗑️  Tapahtuma poistettu onnistuneesti',
        'error_required_packages': '❌ Vaadittuja paketteja ei ole asennettu: {error}',
        'error_run_install': 'Suorita: pip install -r requirements.txt',
        'error_credentials_missing': '❌ iCloud-tunnuksia ei ole määritetty',
        'error_auth_failed': '❌ Todennus epäonnistui: virheelliset tunnukset',
        'error_network': '❌ Verkkovirhe',
        'error_caldav': '❌ CalDAV-virhe',
        'error_unexpected': '❌ Odottamaton virhe: {type}',
        'error_calendars_not_found': '❌ Kalentereita ei löytynyt',
        'error_calendar_not_found': '❌ Kalenteria \'{name}\' ei löytynyt',
        'error_events': '❌ Virhe tapahtumien haussa',
        'error_listing_calendars': '❌ Virhe kalenterien listauksessa',
        'error_invalid_calendar_name': '❌ Virheellinen kalenterin nimi',
        'error_days_range': '❌ days_ahead täytyy olla välillä {min} - {max}',
        'error_missing_fields': '❌ Pakolliset kentät puuttuvat: {fields}',
        'error_dtstart_type': '❌ dtstart täytyy olla datetime-objekti',
        'error_dtend_type': '❌ dtend täytyy olla datetime-objekti',
        'error_time_range': '❌ Tapahtuman päättymisajan täytyy olla alkamisajan jälkeen',
        'error_creating_event': '❌ Virhe tapahtuman luonnissa',
        'error_event_not_found': '❌ Tapahtumaa tai kalenteria ei löytynyt',
        'error_deleting_event': '❌ Virhe tapahtuman poistossa',
        'error_invalid_uid': '❌ Kelvollinen tapahtuman UID vaaditaan',
        'error_uid_too_long': '❌ Virheellinen tapahtuman UID (liian pitkä)',
        'error_calendar_required': '❌ Kalenterin nimi vaaditaan',
        'error_json_required': '❌ Kalenteri ja JSON-data vaaditaan',
        'error_json_read': '❌ JSON-tiedoston lukeminen epäonnistui',
        'error_json_too_large': '❌ JSON-data liian suuri',
        'error_invalid_json': '❌ Virheellinen JSON: {error}',
        'error_invalid_datetime': '❌ Virheellinen päivämäärä/aika-muoto: {error}',
        'error_uid_required': '❌ Kalenteri ja tapahtuman UID vaaditaan',
        'warning_conflicts': '⚠️  Varoitus: {count} ristiriitainen tapahtuma(a) löydetty:',
        'warning_conflict_item': '   - {summary} ({start} - {end})',
        'confirm_continue': 'Jatka silti? (k/e): ',
        'creation_cancelled': 'Tapahtuman luonti peruutettu',
        'input_timeout': '⏱️  Syötteen aikakatkaisu',
        'using_cached_connection': 'Käytetään välimuistissa olevaa CalDAV-yhteyttä',
        'connection_successful': 'Yhdistetty onnistuneesti iCloud CalDAV:iin',
        'found_calendar': 'Löydetty kalenteri: {name}',
        'found_event': 'Löydetty tapahtuma: {summary}',
        'loaded_from_keyring': 'Salasana ladattu avainnipusta',
    },
    
    'da': {
        'setup_title': '🔧 iCalendar Sync Opsætning',
        'setup_intro': 'For at bruge iCalendar Sync skal du konfigurere dine iCloud-legitimationsoplysninger.',
        'setup_warning': '⚠️  Brug en App-specifik adgangskode, IKKE din almindelige Apple ID-adgangskode.',
        'setup_url': 'Hent den fra: https://appleid.apple.com -> Log ind og sikkerhed -> App-specifikke adgangskoder',
        'email_prompt': '📧 iCloud E-mail: ',
        'password_prompt': '🔑 App-specifik adgangskode (xxxx-xxxx-xxxx-xxxx): ',
        'email_empty': '❌ E-mail kan ikke være tom',
        'password_empty': '❌ Adgangskode kan ikke være tom',
        'invalid_email': '❌ Ugyldigt e-mailformat',
        'continue_anyway': 'Fortsæt alligevel? (j/n): ',
        'password_unusual': '⚠️  Adgangskodeformat ser usædvanligt ud',
        'are_you_sure': 'Er du sikker på, at dette er korrekt? (j/n): ',
        'setup_cancelled': 'Opsætning annulleret',
        'credentials_saved_keyring': '✅ Legitimationsoplysninger gemt sikkert i systemnøglering',
        'keyring_fallback': '⚠️  Kunne ikke få adgang til systemnøglering, bruger .env-fil',
        'credentials_saved_file': '✅ Konfiguration gemt sikkert i {path}',
        'ready_to_use': '🚀 Du kan nu bruge iCalendar Sync!',
        'available_calendars': '📅 Tilgængelige Kalendere ({count}):',
        'events_in_calendar': '📋 Begivenheder i \'{name}\' ({count} fundet):',
        'event_created': '✅ Begivenhed \'{summary}\' oprettet med succes',
        'event_deleted': '🗑️  Begivenhed slettet med succes',
        'error_required_packages': '❌ Påkrævede pakker ikke installeret: {error}',
        'error_run_install': 'Kør: pip install -r requirements.txt',
        'error_credentials_missing': '❌ iCloud-legitimationsoplysninger ikke konfigureret',
        'error_auth_failed': '❌ Godkendelse mislykkedes: ugyldige legitimationsoplysninger',
        'error_network': '❌ Netværksfejl',
        'error_caldav': '❌ CalDAV-fejl',
        'error_unexpected': '❌ Uventet fejl: {type}',
        'error_calendars_not_found': '❌ Kalendere ikke fundet',
        'error_calendar_not_found': '❌ Kalender \'{name}\' ikke fundet',
        'error_events': '❌ Fejl ved hentning af begivenheder',
        'error_listing_calendars': '❌ Fejl ved visning af kalendere',
        'error_invalid_calendar_name': '❌ Ugyldigt kalendernavn',
        'error_days_range': '❌ days_ahead skal være mellem {min} og {max}',
        'error_missing_fields': '❌ Manglende påkrævede felter: {fields}',
        'error_dtstart_type': '❌ dtstart skal være et datetime-objekt',
        'error_dtend_type': '❌ dtend skal være et datetime-objekt',
        'error_time_range': '❌ Begivenhedens sluttid skal være efter starttid',
        'error_creating_event': '❌ Fejl ved oprettelse af begivenhed',
        'error_event_not_found': '❌ Begivenhed eller kalender ikke fundet',
        'error_deleting_event': '❌ Fejl ved sletning af begivenhed',
        'error_invalid_uid': '❌ Gyldig begivenheds-UID påkrævet',
        'error_uid_too_long': '❌ Ugyldig begivenheds-UID (for lang)',
        'error_calendar_required': '❌ Kalendernavn påkrævet',
        'error_json_required': '❌ Kalender og JSON-data påkrævet',
        'error_json_read': '❌ Kunne ikke læse JSON-fil',
        'error_json_too_large': '❌ JSON-data for stor',
        'error_invalid_json': '❌ Ugyldig JSON: {error}',
        'error_invalid_datetime': '❌ Ugyldigt dato/tid-format: {error}',
        'error_uid_required': '❌ Kalender og begivenheds-UID påkrævet',
        'warning_conflicts': '⚠️  Advarsel: {count} konfliktende begivenhed(er) fundet:',
        'warning_conflict_item': '   - {summary} ({start} til {end})',
        'confirm_continue': 'Fortsæt alligevel? (j/n): ',
        'creation_cancelled': 'Oprettelse af begivenhed annulleret',
        'input_timeout': '⏱️  Input-timeout',
        'using_cached_connection': 'Bruger cachet CalDAV-forbindelse',
        'connection_successful': 'Forbundet med succes til iCloud CalDAV',
        'found_calendar': 'Fandt kalender: {name}',
        'found_event': 'Fandt begivenhed: {summary}',
        'loaded_from_keyring': 'Adgangskode indlæst fra nøglering',
    },
    
    'uk': {
        'setup_title': '🔧 Налаштування iCalendar Sync',
        'setup_intro': 'Для використання iCalendar Sync потрібно налаштувати облікові дані iCloud.',
        'setup_warning': '⚠️  Використовуйте пароль для застосунку, НЕ звичайний пароль Apple ID.',
        'setup_url': 'Отримати тут: https://appleid.apple.com -> Вхід та безпека -> Паролі застосунків',
        'email_prompt': '📧 Email iCloud: ',
        'password_prompt': '🔑 Пароль для застосунку (xxxx-xxxx-xxxx-xxxx): ',
        'email_empty': '❌ Email не може бути порожнім',
        'password_empty': '❌ Пароль не може бути порожнім',
        'invalid_email': '❌ Невірний формат email',
        'continue_anyway': 'Продовжити все одно? (y/n): ',
        'password_unusual': '⚠️  Формат пароля виглядає незвично',
        'are_you_sure': 'Ви впевнені, що це правильно? (y/n): ',
        'setup_cancelled': 'Налаштування скасовано',
        'credentials_saved_keyring': '✅ Облікові дані безпечно збережено у системній зв\'язці ключів',
        'keyring_fallback': '⚠️  Не вдалося отримати доступ до зв\'язки ключів, використовується файл .env',
        'credentials_saved_file': '✅ Конфігурація безпечно збережена в {path}',
        'ready_to_use': '🚀 Тепер ви можете використовувати iCalendar Sync!',
        'available_calendars': '📅 Доступні календарі ({count}):',
        'events_in_calendar': '📋 Події в \'{name}\' (знайдено {count}):',
        'event_created': '✅ Подія \'{summary}\' успішно створена',
        'event_deleted': '🗑️  Подія успішно видалена',
        'error_required_packages': '❌ Необхідні пакети не встановлено: {error}',
        'error_run_install': 'Виконайте: pip install -r requirements.txt',
        'error_credentials_missing': '❌ Облікові дані iCloud не налаштовано',
        'error_auth_failed': '❌ Помилка автентифікації: невірні облікові дані',
        'error_network': '❌ Помилка мережі',
        'error_caldav': '❌ Помилка CalDAV',
        'error_unexpected': '❌ Неочікувана помилка: {type}',
        'error_calendars_not_found': '❌ Календарі не знайдено',
        'error_calendar_not_found': '❌ Календар \'{name}\' не знайдено',
        'error_events': '❌ Помилка отримання подій',
        'error_listing_calendars': '❌ Помилка отримання списку календарів',
        'error_invalid_calendar_name': '❌ Невірне ім\'я календаря',
        'error_days_range': '❌ days_ahead має бути від {min} до {max}',
        'error_missing_fields': '❌ Відсутні обов\'язкові поля: {fields}',
        'error_dtstart_type': '❌ dtstart має бути об\'єктом datetime',
        'error_dtend_type': '❌ dtend має бути об\'єктом datetime',
        'error_time_range': '❌ Час закінчення події має бути після часу початку',
        'error_creating_event': '❌ Помилка створення події',
        'error_event_not_found': '❌ Подія або календар не знайдені',
        'error_deleting_event': '❌ Помилка видалення події',
        'error_invalid_uid': '❌ Потрібен коректний UID події',
        'error_uid_too_long': '❌ Невірний UID події (занадто довгий)',
        'error_calendar_required': '❌ Потрібне ім\'я календаря',
        'error_json_required': '❌ Потрібні календар та дані JSON',
        'error_json_read': '❌ Не вдалося прочитати файл JSON',
        'error_json_too_large': '❌ Дані JSON занадто великі',
        'error_invalid_json': '❌ Невірний JSON: {error}',
        'error_invalid_datetime': '❌ Невірний формат дати/часу: {error}',
        'error_uid_required': '❌ Потрібні календар та UID події',
        'warning_conflicts': '⚠️  Попередження: знайдено {count} конфліктуючих подій:',
        'warning_conflict_item': '   - {summary} ({start} до {end})',
        'confirm_continue': 'Продовжити все одно? (y/n): ',
        'creation_cancelled': 'Створення події скасовано',
        'input_timeout': '⏱️  Час очікування введення минув',
        'using_cached_connection': 'Використовується кешоване підключення CalDAV',
        'connection_successful': 'Успішно підключено до iCloud CalDAV',
        'found_calendar': 'Знайдено календар: {name}',
        'found_event': 'Знайдено подію: {summary}',
        'loaded_from_keyring': 'Пароль завантажено зі зв\'язки ключів',
    },
}

# Add existing translations for es, fr, de, it from previous version
# (keeping them as they were already complete)

TRANSLATIONS['es'] = {
    'setup_title': '🔧 Configuración de iCalendar Sync',
    'setup_intro': 'Para usar iCalendar Sync, necesitas configurar tus credenciales de iCloud.',
    'setup_warning': '⚠️  Usa una Contraseña Específica de App, NO tu contraseña regular de Apple ID.',
    'setup_url': 'Obténla de: https://appleid.apple.com -> Inicio de sesión y seguridad -> Contraseñas específicas de apps',
    'email_prompt': '📧 Email de iCloud: ',
    'password_prompt': '🔑 Contraseña Específica de App (xxxx-xxxx-xxxx-xxxx): ',
    'email_empty': '❌ El email no puede estar vacío',
    'password_empty': '❌ La contraseña no puede estar vacía',
    'invalid_email': '❌ Formato de email inválido',
    'continue_anyway': '¿Continuar de todos modos? (s/n): ',
    'password_unusual': '⚠️  El formato de contraseña parece inusual',
    'are_you_sure': '¿Estás seguro de que es correcto? (s/n): ',
    'setup_cancelled': 'Configuración cancelada',
    'credentials_saved_keyring': '✅ Credenciales guardadas de forma segura en el llavero del sistema',
    'keyring_fallback': '⚠️  No se pudo acceder al llavero del sistema, usando archivo .env',
    'credentials_saved_file': '✅ Configuración guardada de forma segura en {path}',
    'ready_to_use': '🚀 ¡Ya puedes usar iCalendar Sync!',
    'available_calendars': '📅 Calendarios Disponibles ({count}):',
    'events_in_calendar': '📋 Eventos en \'{name}\' ({count} encontrados):',
    'event_created': '✅ Evento \'{summary}\' creado exitosamente',
    'event_deleted': '🗑️  Evento eliminado exitosamente',
    'error_required_packages': '❌ Paquetes requeridos no instalados: {error}',
    'error_run_install': 'Ejecuta: pip install -r requirements.txt',
    'error_credentials_missing': '❌ Credenciales de iCloud no configuradas',
    'error_auth_failed': '❌ Autenticación fallida: Credenciales inválidas',
    'error_network': '❌ Error de red',
    'error_caldav': '❌ Error de CalDAV',
    'error_unexpected': '❌ Error inesperado: {type}',
    'error_calendars_not_found': '❌ Calendarios no encontrados',
    'error_calendar_not_found': '❌ Calendario \'{name}\' no encontrado',
    'error_events': '❌ Error obteniendo eventos',
    'error_listing_calendars': '❌ Error listando calendarios',
    'error_invalid_calendar_name': '❌ Nombre de calendario inválido',
    'error_days_range': '❌ days_ahead debe estar entre {min} y {max}',
    'error_missing_fields': '❌ Campos requeridos faltantes: {fields}',
    'error_dtstart_type': '❌ dtstart debe ser un objeto datetime',
    'error_dtend_type': '❌ dtend debe ser un objeto datetime',
    'error_time_range': '❌ La hora de fin del evento debe ser posterior a la de inicio',
    'error_creating_event': '❌ Error creando evento',
    'error_event_not_found': '❌ Evento o calendario no encontrado',
    'error_deleting_event': '❌ Error eliminando evento',
    'error_invalid_uid': '❌ Se requiere un UID de evento válido',
    'error_uid_too_long': '❌ UID de evento inválido (demasiado largo)',
    'error_calendar_required': '❌ Se requiere nombre de calendario',
    'error_json_required': '❌ Se requieren calendario y datos JSON',
    'error_json_read': '❌ No se pudo leer el archivo JSON',
    'error_json_too_large': '❌ Datos JSON demasiado grandes',
    'error_invalid_json': '❌ JSON inválido: {error}',
    'error_invalid_datetime': '❌ Formato de fecha/hora inválido: {error}',
    'error_uid_required': '❌ Se requieren calendario y UID de evento',
    'warning_conflicts': '⚠️  Advertencia: {count} evento(s) en conflicto encontrado(s):',
    'warning_conflict_item': '   - {summary} ({start} a {end})',
    'confirm_continue': '¿Continuar de todos modos? (s/n): ',
    'creation_cancelled': 'Creación de evento cancelada',
    'input_timeout': '⏱️  Tiempo de espera agotado',
    'using_cached_connection': 'Usando conexión CalDAV en caché',
    'connection_successful': 'Conectado exitosamente a iCloud CalDAV',
    'found_calendar': 'Calendario encontrado: {name}',
    'found_event': 'Evento encontrado: {summary}',
    'loaded_from_keyring': 'Contraseña cargada desde el llavero',
}

# French, German, Italian translations remain the same (omitted for brevity - they're already in the file)

def get_language() -> str:
    """
    Detect system language or use environment variable
    Priority: ICALENDAR_LANG > system locale > English
    """
    env_lang = os.getenv('ICALENDAR_LANG', '').lower()
    if env_lang in SUPPORTED_LANGUAGES:
        return env_lang
    
    try:
        system_locale = locale.getdefaultlocale()[0]
        if system_locale:
            lang_code = system_locale.split('_')[0].lower()
            if lang_code in SUPPORTED_LANGUAGES:
                return lang_code
    except Exception:
        pass
    
    return 'en'

CURRENT_LANGUAGE = get_language()

def _(key: str, **kwargs) -> str:
    """
    Get translated string for current language
    
    Args:
        key: Translation key
        **kwargs: Format parameters
    
    Returns:
        Translated and formatted string
    """
    translations = TRANSLATIONS.get(CURRENT_LANGUAGE, TRANSLATIONS['en'])
    text = translations.get(key, TRANSLATIONS['en'].get(key, key))
    
    if kwargs:
        try:
            return text.format(**kwargs)
        except (KeyError, ValueError):
            return text
    
    return text

def set_language(lang: str) -> bool:
    """
    Set language manually
    
    Args:
        lang: Language code (en, es, fr, de, it, ru, nl, sv, fi, da, uk)
    
    Returns:
        True if language was set, False if invalid
    """
    global CURRENT_LANGUAGE
    
    lang = lang.lower()
    if lang in SUPPORTED_LANGUAGES:
        CURRENT_LANGUAGE = lang
        return True
    
    return False

def get_supported_languages() -> list:
    """Get list of supported language codes"""
    return SUPPORTED_LANGUAGES.copy()
