#!/usr/bin/env bash
#
# iblipper.sh - Generate iBlipper URLs from the command line
#
# Usage:
#   iblipper.sh "Your Message"                    # Default (emphatic, dark)
#   iblipper.sh "Your Message" excited            # With emotion
#   iblipper.sh "Your Message" casual light       # With emotion + light mode
#   iblipper.sh --gif "Your Message" emphatic     # GIF export URL
#
# Outputs a ready-to-use URL or markdown link.

set -e

BASE_URL="https://andyed.github.io/iblipper2025/"

show_help() {
    cat << 'EOF'
iblipper.sh - Generate iBlipper kinetic typography URLs

USAGE:
    iblipper.sh [OPTIONS] "MESSAGE" [EMOTION] [MODE]

OPTIONS:
    --gif           Output GIF export URL (requires browser to download)
    --markdown, -m  Output as markdown link with ▶️ prefix
    --help, -h      Show this help

ARGUMENTS:
    MESSAGE         Text to animate (required)
    EMOTION         Animation style (default: emphatic)
                    Options: neutral, hurry, idyllic, question, response_required,
                             excited, playful, emphatic, casual, electric, wobbly
    MODE            dark (default) or light

EXAMPLES:
    iblipper.sh "Hello World"
    iblipper.sh "Breaking News" emphatic
    iblipper.sh "Good morning" casual light
    iblipper.sh -m "Click me!" excited
    iblipper.sh --gif "Export this" playful

EOF
}

# Parse flags
GIF_MODE=false
MARKDOWN=false

while [[ $# -gt 0 ]]; do
    case "$1" in
        --gif)
            GIF_MODE=true
            shift
            ;;
        --markdown|-m)
            MARKDOWN=true
            shift
            ;;
        --help|-h)
            show_help
            exit 0
            ;;
        -*)
            echo "Unknown option: $1" >&2
            exit 1
            ;;
        *)
            break
            ;;
    esac
done

# Get arguments
MESSAGE="${1:-}"
EMOTION="${2:-emphatic}"
MODE="${3:-dark}"

if [[ -z "$MESSAGE" ]]; then
    echo "Error: Message is required" >&2
    echo "Usage: iblipper.sh \"Your Message\" [emotion] [dark|light]" >&2
    exit 1
fi

# URL encode the message (spaces as +, special chars encoded)
encode_text() {
    local text="$1"
    python3 -c "import urllib.parse; print(urllib.parse.quote_plus('''$text'''))"
}

ENCODED_TEXT=$(encode_text "$MESSAGE")

# Determine dark mode
DARK_BOOL="true"
[[ "$MODE" == "light" ]] && DARK_BOOL="false"

# Build URL
if $GIF_MODE; then
    URL="${BASE_URL}?export=gif#text=${ENCODED_TEXT}&emotion=${EMOTION}&dark=${DARK_BOOL}"
else
    URL="${BASE_URL}#text=${ENCODED_TEXT}&emotion=${EMOTION}&dark=${DARK_BOOL}&share=yes"
fi

# Output
if $MARKDOWN; then
    echo "[▶️ ${MESSAGE}](${URL})"
else
    echo "$URL"
fi
