#!/usr/bin/env python3
"""Auto-replace common AI patterns in text."""
import argparse, json, re, sys
from pathlib import Path

SCRIPT_DIR = Path(__file__).parent
PATTERNS = json.loads((SCRIPT_DIR / "patterns.json").read_text())

def replace_phrases(text: str) -> tuple[str, list[str]]:
    changes = []
    for old, new in PATTERNS["replacements"].items():
        if " " in old or old.endswith(","):
            pattern = re.compile(re.escape(old), re.IGNORECASE)
        else:
            pattern = re.compile(r"\b" + re.escape(old) + r"\b", re.IGNORECASE)
        matches = pattern.findall(text)
        if matches:
            if new:
                changes.append(f'"{old}" → "{new}" ({len(matches)}x)')
            else:
                changes.append(f'"{old}" → (removed) ({len(matches)}x)')
            text = pattern.sub(new, text)
    return text, changes

def fix_curly_quotes(text: str) -> tuple[str, bool]:
    original = text
    text = text.replace(""", '"').replace(""", '"')
    text = text.replace("'", "'").replace("'", "'")
    return text, text != original

def remove_chatbot_artifacts(text: str) -> tuple[str, list[str]]:
    changes = []
    for artifact in PATTERNS["chatbot_artifacts"]:
        pattern = re.compile(r"[^.!?\n]*" + re.escape(artifact) + r"[^.!?\n]*[.!?]?\s*", re.IGNORECASE)
        if pattern.search(text):
            changes.append(f'Removed sentence with "{artifact}"')
            text = pattern.sub("", text)
    return text, changes

def reduce_em_dashes(text: str) -> tuple[str, int]:
    count = text.count("—")
    text = re.sub(r"\s*—\s*", ", ", text)
    text = text.replace("--", ", ")
    return text, count

def clean_whitespace(text: str) -> str:
    text = re.sub(r" +", " ", text)
    text = re.sub(r"\n{3,}", "\n\n", text)
    text = re.sub(r"^\s+", "", text, flags=re.MULTILINE)
    return text.strip()

def fix_capitalization(text: str) -> str:
    text = re.sub(r"(^|[.!?]\s+)([a-z])", lambda m: m.group(1) + m.group(2).upper(), text)
    text = re.sub(r"^\s*([a-z])", lambda m: m.group(1).upper(), text)
    return text

def humanize(text: str, fix_dashes: bool = False, remove_artifacts: bool = True) -> tuple[str, list[str]]:
    all_changes = []
    
    text, changes = replace_phrases(text)
    all_changes.extend(changes)
    
    text, fixed = fix_curly_quotes(text)
    if fixed:
        all_changes.append("Fixed curly quotes → straight quotes")
    
    if remove_artifacts:
        text, changes = remove_chatbot_artifacts(text)
        all_changes.extend(changes)
    
    if fix_dashes:
        text, count = reduce_em_dashes(text)
        if count:
            all_changes.append(f"Replaced {count} em dashes with commas")
    
    text = clean_whitespace(text)
    text = fix_capitalization(text)
    
    return text, all_changes

def main():
    parser = argparse.ArgumentParser(description="Auto-humanize text")
    parser.add_argument("input", nargs="?", help="Input file (or stdin)")
    parser.add_argument("-o", "--output", help="Output file (default: stdout)")
    parser.add_argument("--fix-dashes", action="store_true", help="Replace em dashes with commas")
    parser.add_argument("--keep-artifacts", action="store_true", help="Don't remove chatbot artifacts")
    parser.add_argument("--quiet", "-q", action="store_true", help="Only output result, no changes log")
    args = parser.parse_args()
    
    if args.input:
        text = Path(args.input).read_text()
    else:
        text = sys.stdin.read()
    
    result, changes = humanize(
        text, 
        fix_dashes=args.fix_dashes,
        remove_artifacts=not args.keep_artifacts
    )
    
    if not args.quiet and changes:
        print("CHANGES MADE:", file=sys.stderr)
        for c in changes:
            print(f"  • {c}", file=sys.stderr)
        print(file=sys.stderr)
    
    if args.output:
        Path(args.output).write_text(result)
        if not args.quiet:
            print(f"Written to {args.output}", file=sys.stderr)
    else:
        print(result)

if __name__ == "__main__":
    main()
