#!/usr/bin/env python3
"""Analyze text for AI writing patterns."""
import argparse, json, re, sys
from pathlib import Path
from collections import Counter

SCRIPT_DIR = Path(__file__).parent
PATTERNS = json.loads((SCRIPT_DIR / "patterns.json").read_text())

def find_matches(text: str, patterns: list[str]) -> list[tuple[str, int]]:
    matches = []
    text_lower = text.lower()
    for p in patterns:
        count = text_lower.count(p.lower())
        if count > 0:
            matches.append((p, count))
    return sorted(matches, key=lambda x: -x[1])

def count_em_dashes(text: str) -> int:
    return text.count("—") + text.count("--")

def count_curly_quotes(text: str) -> int:
    return len(re.findall(r'[""]', text))

def analyze(text: str) -> dict:
    results = {
        "ai_words": find_matches(text, PATTERNS["ai_words"]),
        "puffery": find_matches(text, PATTERNS["puffery"]),
        "chatbot_artifacts": find_matches(text, PATTERNS["chatbot_artifacts"]),
        "hedging": find_matches(text, PATTERNS["hedging_phrases"]),
        "em_dashes": count_em_dashes(text),
        "curly_quotes": count_curly_quotes(text),
        "replaceable": [(k, text.lower().count(k.lower())) 
                        for k, v in PATTERNS["replacements"].items() 
                        if k.lower() in text.lower()],
    }
    results["total_issues"] = (
        sum(c for _, c in results["ai_words"]) +
        sum(c for _, c in results["puffery"]) +
        sum(c for _, c in results["chatbot_artifacts"]) +
        sum(c for _, c in results["hedging"]) +
        results["em_dashes"] +
        results["curly_quotes"] +
        sum(c for _, c in results["replaceable"])
    )
    return results

def print_report(results: dict):
    print(f"\n{'='*50}")
    print(f"AI PATTERN ANALYSIS - {results['total_issues']} issues found")
    print(f"{'='*50}\n")
    
    if results["ai_words"]:
        print("AI VOCABULARY:")
        for word, count in results["ai_words"]:
            print(f"  • {word}: {count}x")
        print()
    
    if results["puffery"]:
        print("PUFFERY/PROMOTIONAL:")
        for word, count in results["puffery"]:
            print(f"  • {word}: {count}x")
        print()
    
    if results["chatbot_artifacts"]:
        print("CHATBOT ARTIFACTS:")
        for phrase, count in results["chatbot_artifacts"]:
            print(f"  • \"{phrase}\": {count}x")
        print()
    
    if results["hedging"]:
        print("EXCESSIVE HEDGING:")
        for phrase, count in results["hedging"]:
            print(f"  • \"{phrase}\": {count}x")
        print()
    
    if results["replaceable"]:
        print("AUTO-REPLACEABLE:")
        for phrase, count in results["replaceable"]:
            repl = PATTERNS["replacements"][phrase]
            arrow = f" → \"{repl}\"" if repl else " → (remove)"
            print(f"  • \"{phrase}\"{arrow}: {count}x")
        print()
    
    if results["em_dashes"] > 2:
        print(f"EM DASHES: {results['em_dashes']} (consider reducing)\n")
    
    if results["curly_quotes"]:
        print(f"CURLY QUOTES: {results['curly_quotes']} (replace with straight quotes)\n")
    
    if results["total_issues"] == 0:
        print("✓ No obvious AI patterns detected.\n")

def main():
    parser = argparse.ArgumentParser(description="Analyze text for AI patterns")
    parser.add_argument("input", nargs="?", help="Input file (or stdin)")
    parser.add_argument("--json", action="store_true", help="Output as JSON")
    args = parser.parse_args()
    
    if args.input:
        text = Path(args.input).read_text()
    else:
        text = sys.stdin.read()
    
    results = analyze(text)
    
    if args.json:
        print(json.dumps(results, indent=2))
    else:
        print_report(results)

if __name__ == "__main__":
    main()
