#!/usr/bin/env bash
set -euo pipefail

# Build/update references/naming-context.md with entity naming hints.
#
# Usage:
#   ./scripts/save_naming_context.sh
#   ./scripts/save_naming_context.sh /custom/path/naming-context.md

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
CALL="$SCRIPT_DIR/ha_call.sh"
OUT_FILE="${1:-$SCRIPT_DIR/../references/naming-context.md}"

if ! command -v jq >/dev/null 2>&1; then
  echo "Error: jq is required. Install jq and try again." >&2
  exit 1
fi

TMP_JSON="$(mktemp)"
TMP_AUTO="$(mktemp)"
trap 'rm -f "$TMP_JSON" "$TMP_AUTO"' EXIT

"$CALL" GET /api/states > "$TMP_JSON"

jq -r '
  map(select(.entity_id != null))
  | sort_by(.entity_id)
  | .[]
  | [
      (.entity_id),
      (.entity_id | gsub("[_\\.]"; " ")),
      ((.attributes.friendly_name // "") | ascii_downcase),
      (.entity_id | split(".")[0])
    ]
  | "- " + (.[0]) + "\n  - normalized: " + (.[1]) +
    (if .[2] != "" then "\n  - friendly: " + .[2] else "" end) +
    "\n  - domain: " + .[3] + "\n"
' "$TMP_JSON" > "$TMP_AUTO"

if [[ ! -f "$OUT_FILE" ]]; then
  cat > "$OUT_FILE" <<'EOF'
# Home Assistant Naming Context

Use this file to keep stable, user-specific naming mappings so commands like
"turn on living room light" resolve reliably.

## How to use

1. Run `./scripts/save_naming_context.sh` to refresh auto-detected candidates.
2. Keep `## Manual aliases` as the source of truth for your preferred names.
3. Add multiple aliases for the same entity if needed.

## Manual aliases (authoritative)

- example alias => light.example_living_room

## Auto-detected candidates

<!-- generated by scripts/save_naming_context.sh -->
EOF
fi

awk '
  BEGIN {seen=0}
  {print}
  /^<!-- generated by scripts\/save_naming_context\.sh -->$/ {seen=1; exit}
' "$OUT_FILE" > "$OUT_FILE.tmp"

{
  cat "$OUT_FILE.tmp"
  echo
  echo "Generated on $(date '+%Y-%m-%d %H:%M:%S %Z')"
  echo
  cat "$TMP_AUTO"
} > "$OUT_FILE"

rm -f "$OUT_FILE.tmp"

echo "Updated naming context: $OUT_FILE"
