---
name: hodlxxi-bitcoin-identity
version: 1.0.0
description: Integrate HODLXXI as a Bitcoin-native identity provider that bridges OAuth2/OIDC and Lightning LNURL-Auth for client registration, authorization flows, JWT verification, and health monitoring.
homepage: https://github.com/hodlxxi/Universal-Bitcoin-Identity-Layer
metadata:
  category: authentication
  license: MIT
  tags:
    - oauth2
    - oidc
    - lnurl-auth
    - jwt
    - bitcoin
  dependencies:
    - curl
    - python
    - ecdsa
    - pyjwt
    - requests
---

# HODLXXI Bitcoin Identity

## Overview

Use this skill to integrate HODLXXI (Universal Bitcoin Identity Layer) for agent authentication, LNURL-Auth linking, and JWT-based identity claims.

## Installation

1. Fetch the skill file from the repository (raw link works for installable agents):

```bash
curl -L -o SKILL.md \
  https://raw.githubusercontent.com/hodlxxi/Universal-Bitcoin-Identity-Layer/main/skills/public/hodlxxi-bitcoin-identity/SKILL.md
```

2. Install helper dependencies for local verification scripts:

```bash
python -m pip install ecdsa pyjwt requests
```

## Quick start

1. Set a base URL for the HODLXXI deployment.
2. Register an OAuth client to obtain `client_id` and `client_secret`.
3. Run the OAuth2/OIDC authorization code flow (PKCE recommended).
4. Start an LNURL-Auth session for Lightning wallet login.
5. Verify JWTs with the JWKS endpoint.

## Usage steps

### 1) Configure the base URL

Set the base URL to the HODLXXI deployment (update as needed):

```bash
BASE_URL="https://hodlxxi.com"
```

### 2) Register an OAuth client

Register a client to get credentials:

```bash
curl -X POST "$BASE_URL/oauth/register" \
  -H "Content-Type: application/json" \
  -d '{"client_name": "YourAgentName", "redirect_uris": ["https://your-callback-url"], "scopes": ["openid", "profile"]}'
```

Store `client_id` and `client_secret` securely.

### 3) Run OAuth2/OIDC authorization code flow

Discover endpoints:

```bash
curl "$BASE_URL/.well-known/openid-configuration"
```

Create an authorization request (PKCE recommended):

```bash
curl "$BASE_URL/oauth/authorize?client_id=your_client_id&redirect_uri=your_callback&response_type=code&scope=openid%20profile&code_challenge=your_challenge&code_challenge_method=S256"
```

Exchange the authorization code for tokens:

```bash
curl -X POST "$BASE_URL/oauth/token" \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "grant_type=authorization_code&code=received_code&redirect_uri=your_callback&client_id=your_client_id&code_verifier=your_verifier"
```

Expect an access token, ID token (JWT), and optional refresh token.

### 4) Start an LNURL-Auth session

Create a session and show the LNURL to the user:

```bash
curl -X POST "$BASE_URL/api/lnurl-auth/create" \
  -H "Accept: application/json"
```

Poll for completion after the user scans the LNURL with a Lightning wallet:

```bash
curl "$BASE_URL/api/lnurl-auth/check/your_session_id"
```

### 5) Verify JWTs

Fetch JWKS:

```bash
curl "$BASE_URL/oauth/jwks.json"
```

Verify with Python (example uses PyJWT):

```python
import jwt
import requests

jwks = requests.get("https://your-hodlxxi-deployment.com/oauth/jwks.json", timeout=10).json()
public_key = jwt.algorithms.RSAAlgorithm.from_jwk(jwks["keys"][0])
claims = jwt.decode(your_jwt, public_key, algorithms=["RS256"], audience="your_audience")
print(claims)
```

### 6) Monitor health and metrics

Check liveness and OAuth system status endpoints:

```bash
curl "$BASE_URL/health"
curl "$BASE_URL/oauthx/status"
```

## Code examples

### Register a client from a JSON template

```bash
curl -X POST "$BASE_URL/oauth/register" \
  -H "Content-Type: application/json" \
  -d @templates/oauth-client.json
```

### Create LNURL session and poll

```bash
session_json=$(curl -s -X POST "$BASE_URL/api/lnurl-auth/create")
session_id=$(python3 -c 'import json,sys; print(json.loads(sys.argv[1])["session_id"])' "$session_json")
curl "$BASE_URL/api/lnurl-auth/check/$session_id"
```

## Best practices

- Always use HTTPS and verify TLS certificates in production.
- Keep client secrets in a secrets manager or environment variables.
- Use PKCE for public clients and rotate secrets for confidential clients.
- Treat LNURL sessions as single-use and enforce short TTLs.
- Validate `aud`, `iss`, and `exp` claims for JWTs.

## Advanced features

- Use `/oauthx/docs` for live OAuth/OIDC API documentation.
- Use `/oauthx/status` to monitor database and LNURL session health.
- Rotate JWKS keys via the server configuration (JWKS directory + rotation days).

## PAYG billing for OAuth clients

Paid API calls are billed per **OAuth `client_id`** (agent/app), not per session pubkey. When balance or free quota is exhausted, paid endpoints return **HTTP 402** with a Lightning top-up path.

### Billing endpoints (OAuth token required)

- `POST /api/billing/agent/create-invoice`
- `POST /api/billing/agent/check-invoice`

Example create invoice:

```bash
curl -X POST "$BASE_URL/api/billing/agent/create-invoice" \
  -H "Authorization: Bearer $ACCESS_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"amount_sats": 1000}'
```

Example check invoice:

```bash
curl -X POST "$BASE_URL/api/billing/agent/check-invoice" \
  -H "Authorization: Bearer $ACCESS_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"invoice_id": "your_invoice_id"}'
```

### 402 response shape

When a paid endpoint is called with insufficient balance, expect:

```json
{
  "ok": false,
  "error": "payment_required",
  "code": "PAYMENT_REQUIRED",
  "client_id": "your_client_id",
  "cost_sats": 1,
  "balance_sats": 0,
  "create_invoice_endpoint": "/api/billing/agent/create-invoice",
  "hint": "Top up via Lightning PAYG"
}
```

## Supporting files

- `scripts/verify_signature.py` validates LNURL-Auth signatures locally.
- `HEARTBEAT.md` describes periodic health checks for the deployment.
- `templates/oauth-client.json` provides a ready client registration payload.

## Optional helper script

Use `scripts/verify_signature.py` to validate LNURL signatures locally. Install the dependency first:

```bash
python -m pip install ecdsa
python scripts/verify_signature.py --k1 <hex> --signature <hex> --pubkey <hex>
```
