#!/usr/bin/env -S uv run --script
# /// script
# requires-python = ">=3.10"
# dependencies = [
#     "requests>=2.32.5",
#     "trafilatura>=2.0.0",
#     "urllib3>=2.6.3",
# ]
# ///

import argparse
import html
import json
import re
import textwrap
from pathlib import Path

import requests
from urllib3 import Retry
from trafilatura import fetch_url, extract


def fmt_json(data) -> dict:
    return json.dumps(data, indent=2, ensure_ascii=False)


def tee_json(data):
    print(fmt_json(data))
    return data


def is_uri_file(uri: str) -> bool:
    """Check if a uri is a file path, either url or path."""
    uri = str(uri).strip()
    # if ends with / it's a directory, not a file
    if uri.endswith('/'):
        return False
    # foo-bar or foo.bar or foo.bar.baz
    basename = uri.rsplit('/', 1)[-1]
    return '.' in basename


def get_request_session(retry_total=3):
    """Get a request session with automatic retry."""
    retries = Retry(total=retry_total)
    adapter = requests.adapters.HTTPAdapter(max_retries=retries)
    session = requests.Session()
    session.mount("https://", adapter)
    return session


request_session = get_request_session(retry_total=3)


def get_json_from_url(url: str) -> dict:
    """Get JSON from a URL with automatic retry."""
    response = request_session.get(url)
    response.raise_for_status()
    return response.json()


def clean_html_text(text):
    """Clean HTML text by replacing <p> tags with newlines and removing all remaining HTML tags."""

    # 1. Replace <p> tags with newlines to preserve paragraph structure
    # This ensures "end.<p>Start" becomes "end.\nStart" instead of "end.Start"
    text = re.sub(r'<p\s*/?>', '\n\n', text, flags=re.IGNORECASE)

    # 2. Remove all remaining HTML tags
    text = re.sub(r'<[^>]+>', '', text)

    # 3. Decode HTML entities (e.g. &quot; -> ", &#x27; -> ')
    # We do this last so that decoded characters like < or > aren't mistaken for tags
    text = html.unescape(text)

    return text.strip()


def save_file(path: Path, content: str | dict) -> Path:
    path = Path(path)
    path.parent.mkdir(parents=True, exist_ok=True)
    if not isinstance(content, str):
        content = fmt_json(content)
    path.write_text(content)
    return path


class HackerNewsExtractor:
    """Extractor article and comments from a HackerNews Post."""

    def __init__(self, data: dict):
        self.data = data
        self.lines = []
        self.indent_char = ' ' * 4
        self.indent_level = 0
        self.split_line = '\n' + '-' * 80

        self.article_html = ""
        self.article_text = ""
        self.content = ""

    @classmethod
    def from_json_file(cls, path) -> 'HackerNewsExtractor':
        assert path.is_file(), f"json file not found: {path}"
        assert path.suffix == ".json", f"input file must be json: {path}"
        json_str = path.read_text()
        data = json.loads(json_str)
        return cls(data)

    @classmethod
    def from_id(cls, id: str) -> 'HackerNewsExtractor':
        id_pattern = r"\b(\d{1,10})\b"
        match = re.match(id_pattern, id)
        if match:
            url = f"https://hn.algolia.com/api/v1/items/{id}"
            data = get_json_from_url(url)
            return cls(data)

    @classmethod
    def from_url(cls, url: str) -> 'HackerNewsExtractor':
        url = url.lower().strip()
        url_pattern = r"https://news.ycombinator.com/item\?id=(\d{1,10})"
        match = re.match(url_pattern, url)
        if match:
            return cls.from_id(match.group(1))

    @classmethod
    def from_uri(cls, uri: str) -> 'HackerNewsExtractor':
        # uri can be id/url/file

        if uri.isdigit() or isinstance(uri, int):
            return cls.from_id(uri)

        path = Path(uri)
        if path.is_file():
            return cls.from_json_file(path)

        return cls.from_url(uri)

    def add_line(self, line: str, indent_level: int = 0, sep="", width=80):
        if line.strip():
            indent = self.indent_char * indent_level
            indent_line = textwrap.indent(
                textwrap.fill(line, width=width) if width else line,
                indent,
            )
            self.lines.append(indent_line + sep)

    def add_paragraph(self, *args, sep="\n", **kwargs):
        return self.add_line(*args, sep=sep, **kwargs)

    def get_html_form_url(self, url: str):
        return fetch_url(url)

    def get_text_from_html(self, html: str):
        return extract(
            html,
            output_format="txt",
            fast=False,
            include_comments=False,
        )

    def extract(self) -> str:
        """Extract text from the origin."""
        assert self.data, "data must be populated before extract"

        id = self.data.get("id", "")
        self.id = id
        self.hn_id = id
        hn_url = f"https://news.ycombinator.com/item?id={id}"
        title = self.data.get("title", "")
        author = self.data.get("author", "")
        created_at = self.data.get("created_at", "")
        points = self.data.get("points", "")
        # story_id = int(self.data.get("story_id", 0))
        article_url = self.data.get("url", "")
        children = self.data.get("children", [])

        self.add_line('---')
        self.add_line(f"title: {title}")
        self.add_line(f"author: {author}")
        self.add_line(f"created_at: {created_at}")
        self.add_line(f"url: {article_url}")
        self.add_line(f"points: {points}")
        self.add_line(f"hn_url: {hn_url}", width=0)
        self.add_line(f"comments: {len(children)}")
        self.add_line('---', sep="\n")

        self.add_paragraph(f"# {title}")

        self.article_html = self.get_html_form_url(article_url)
        self.article_text = self.get_text_from_html(self.article_html)
        for paragraph in self.article_text.splitlines():
            self.add_paragraph(paragraph)

        self.add_line(self.split_line)
        self.add_paragraph("## Comments")
        for child in children:
            # direct child indent at level 0
            self.extract_comment(child, indent_level=0)

        self.content = "\n".join(self.lines)
        return self.content

    def extract_comment(self, comment: dict, indent_level: int = 0):
        author = comment.get("author", "")
        text = comment.get("text", "")
        text = clean_html_text(text)
        self.add_paragraph(f"{author}: {text}", indent_level=indent_level)
        # child comment indent 1 more level
        child_indent_level = indent_level + 1
        for child in comment.get("children", []):
            self.extract_comment(child, indent_level=child_indent_level)

    def output(self, path, verbose=False):
        if path:
            self.output_to_file(path, verbose=verbose)
        else:
            self.output_to_stdout(verbose=verbose)

    def output_to_file(self, path, verbose=False):
        path = Path(path)
        if path.suffix:  # .md or .txt
            out_path = path
            out_dir = path.parent
        else:
            out_dir = path
            out_path = out_dir / f"hn-{self.hn_id}.md"
        save_file(out_path, self.content)
        if verbose:
            save_file(out_path.with_suffix(".json"), self.data)
            save_file(out_path.with_suffix(".html"), self.article_html)
            save_file(out_path.with_suffix(".txt"), self.article_text)

    def output_to_stdout(self, verbose=False):
        print(self.content)


def main():
    parser = argparse.ArgumentParser(
        description="HackerNews Extractor"
    )
    parser.add_argument("uri", type=str, help="HackerNews id, url, or json file path")
    parser.add_argument("-o", "--output", help="output file path, default to stdout")
    parser.add_argument("-v", "--verbose", action="store_true", help="verbose/debug mode")
    args = parser.parse_args()
    extractor = HackerNewsExtractor.from_uri(args.uri)
    if not extractor:
        raise ValueError(f"Invalid HN uri, please provide a valid id, url, or json file path: {args.uri}")

    extractor.extract()
    extractor.output(args.output, verbose=args.verbose)


if __name__ == "__main__":
    main()
