# HL Privateer -- x402 Payment Quickstart

How to pay for HL Privateer agent endpoints using x402 v2.

Part of the skill package at `https://hlprivateer.xyz/skills/`.

## Overview

HL Privateer agent endpoints are gated by x402 v2 (exact scheme). Every paid request follows a three-step flow:

1. **Request** -- Client sends a normal GET request
2. **Challenge** -- Server responds `402 Payment Required` with payment instructions
3. **Pay and Retry** -- Client signs payment and retries with proof

No API keys. No sign-ups. No accounts. Just USDC on Base.

## Payment Details

| Property | Value |
|----------|-------|
| Network | Base (eip155:8453) |
| Asset | USDC |
| Asset Address | `0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913` |
| Facilitator | `https://facilitator.payai.network` |
| Protocol | x402 v2 |
| Scheme | exact |

## Route Pricing

| Endpoint | Price |
|----------|-------|
| `/v1/agent/stream/snapshot` | $0.01 |
| `/v1/agent/positions` | $0.01 |
| `/v1/agent/orders` | $0.01 |
| `/v1/agent/analysis?latest=true` | $0.01 |
| `/v1/agent/analysis` | $0.01 |
| `/v1/agent/insights?scope=market` | $0.02 |
| `/v1/agent/insights?scope=ai` | $0.02 |
| `/v1/agent/copy/trade?kind=signals` | $0.03 |
| `/v1/agent/copy/trade?kind=positions` | $0.03 |

## Payment Flow

### Step 1: Unpaid Request

Send a normal GET request to any agent endpoint.

```bash
curl -i https://api.hlprivateer.xyz/v1/agent/positions
```

### Step 2: Receive 402 Challenge

The server responds with HTTP 402 and a `PAYMENT-REQUIRED` header containing Base64-encoded JSON payment instructions.

```
HTTP/2 402 Payment Required
PAYMENT-REQUIRED: eyJzY2hlbWUiOiJleGFjdCIsIm5ldHdvcmsiOiJlaXAxNTU6ODQ1...
Content-Type: application/json

{
  "error": {
    "code": "PAYMENT_REQUIRED",
    "message": "x402 payment required"
  }
}
```

Decode the `PAYMENT-REQUIRED` header to get payment instructions:

```json
{
  "scheme": "exact",
  "network": "eip155:8453",
  "maxAmountRequired": "10000",
  "resource": "https://api.hlprivateer.xyz/v1/agent/positions",
  "description": "Current open positions with symbols, sides, sizes, entries, PnL",
  "mimeType": "application/json",
  "payTo": "0x...",
  "maxTimeoutSeconds": 300,
  "asset": "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913",
  "extra": {
    "name": "HL Privateer",
    "facilitator": "https://facilitator.payai.network"
  }
}
```

The `maxAmountRequired` field is in the token's smallest unit. For USDC (6 decimals), `10000` = $0.01.

### Step 3: Sign Payment and Retry

Create a signed payment payload and send it as the `PAYMENT-SIGNATURE` header (Base64-encoded JSON).

```bash
curl -i https://api.hlprivateer.xyz/v1/agent/positions \
  -H "PAYMENT-SIGNATURE: eyJ4NDAyVmVyc2lvbiI6MiwicGF5bG9hZCI6ey..."
```

The payment signature payload follows the x402 v2 exact scheme format:

```json
{
  "x402Version": 2,
  "scheme": "exact",
  "payload": {
    "signature": "0x...",
    "authorization": {
      "from": "0x<payer-address>",
      "to": "0x<payTo-address>",
      "value": "10000",
      "validAfter": "0",
      "validBefore": "<unix-timestamp>",
      "nonce": "0x...",
      "version": "2"
    }
  }
}
```

### Step 4: Receive Data

On successful payment verification, the server returns the data along with a `PAYMENT-RESPONSE` header containing settlement confirmation.

```
HTTP/2 200 OK
PAYMENT-RESPONSE: eyJzdWNjZXNzIjp0cnVlLCJ0eEhhc2giOiIweC4uLiJ9
Content-Type: application/json

{
  "positions": [
    {
      "symbol": "HYPE",
      "side": "LONG",
      "size": 12.5,
      "entryPrice": 23.14,
      "unrealizedPnl": 2.18
    }
  ]
}
```

Decode the `PAYMENT-RESPONSE` header for settlement details:

```json
{
  "success": true,
  "txHash": "0x...",
  "network": "eip155:8453"
}
```

## Full Example (curl)

```bash
# Step 1: Get payment challenge
CHALLENGE=$(curl -s -D - https://api.hlprivateer.xyz/v1/agent/positions \
  2>/dev/null | grep -i 'PAYMENT-REQUIRED:' | cut -d' ' -f2)

echo "Challenge: $CHALLENGE"
echo "$CHALLENGE" | base64 -d | jq .

# Step 2: Sign payment (using your x402 client library)
# The signing step depends on your x402 client implementation.
# See https://x402.org for client libraries in TypeScript, Python, etc.

# Step 3: Retry with signed payment
curl -s https://api.hlprivateer.xyz/v1/agent/positions \
  -H "PAYMENT-SIGNATURE: $SIGNED_PAYMENT" | jq .
```

## Using x402 Client Libraries

### TypeScript (@x402/client)

```typescript
import { x402Client } from "@x402/client";
import { createWalletClient, http } from "viem";
import { base } from "viem/chains";
import { privateKeyToAccount } from "viem/accounts";

const account = privateKeyToAccount("0x<your-private-key>");
const wallet = createWalletClient({
  account,
  chain: base,
  transport: http()
});

const client = new x402Client(wallet);

const response = await client.fetch(
  "https://api.hlprivateer.xyz/v1/agent/positions"
);
const data = await response.json();
```

### Python

```python
import httpx
import base64
import json

# Step 1: Get challenge
resp = httpx.get("https://api.hlprivateer.xyz/v1/agent/positions")
assert resp.status_code == 402

challenge = json.loads(
    base64.b64decode(resp.headers["PAYMENT-REQUIRED"])
)

# Step 2: Sign with your x402 library
# payment_signature = sign_x402_payment(challenge, wallet)

# Step 3: Retry
resp = httpx.get(
    "https://api.hlprivateer.xyz/v1/agent/positions",
    headers={"PAYMENT-SIGNATURE": payment_signature}
)
data = resp.json()
```

## Entitlements

After a successful x402 payment, the server may issue an entitlement token in the response. Subsequent requests can include this token to skip re-payment until it expires or quota is exhausted:

```
x-agent-entitlement: <entitlement-id>
```

This is an optimization. If the entitlement is expired or exhausted, the server falls back to the standard 402 challenge flow.

## Header Reference

| Header | Direction | Purpose |
|--------|-----------|---------|
| `PAYMENT-REQUIRED` | Response (402) | Base64 JSON payment instructions |
| `PAYMENT-SIGNATURE` | Request | Base64 JSON signed payment proof |
| `PAYMENT-RESPONSE` | Response (200) | Base64 JSON settlement confirmation |
| `x-agent-entitlement` | Request | Reuse entitlement from prior payment |

## Troubleshooting

| Symptom | Cause | Fix |
|---------|-------|-----|
| 402 on every request | Payment signature invalid or expired | Verify signing key, check timestamp bounds |
| 402 with empty PAYMENT-REQUIRED | Facilitator unreachable | Retry after delay, check facilitator status |
| 403 after payment | Insufficient capability for route | Check tier requirements in the capability gate table |
| 429 | Rate limited | Back off and retry with exponential delay |

## Related Files

- Skill definition: https://hlprivateer.xyz/skills/hl-privateer.md
- Full API reference: https://hlprivateer.xyz/skills/api.md
- Machine-readable discovery: https://hlprivateer.xyz/skills/agents.json
- x402 protocol docs: https://x402.org
